<?php

namespace App;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Chat extends Model
{
    protected $fillable = [
        'name',
        'type'
    ];

    protected $casts = [
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Get the participants for the chat.
     */
    public function participants(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'chat_participants')
                    ->withPivot('last_read_at')
                    ->withTimestamps();
    }

    /**
     * Get the messages for the chat.
     */
    public function messages(): HasMany
    {
        return $this->hasMany(Message::class)->orderBy('created_at', 'asc');
    }

    /**
     * Get the latest message for the chat.
     */
    public function latestMessage(): BelongsTo
    {
        return $this->belongsTo(Message::class, 'id', 'chat_id')
                    ->orderBy('created_at', 'desc');
    }

    /**
     * Check if user is participant of this chat.
     */
    public function hasParticipant($userId): bool
    {
        return $this->participants()->where('user_id', $userId)->exists();
    }

    /**
     * Get other participants (excluding current user).
     */
    public function otherParticipants($currentUserId)
    {
        return $this->participants()->where('user_id', '!=', $currentUserId);
    }

    /**
     * Get chat name for display.
     */
    public function getDisplayName($currentUserId = null): string
    {
        if ($this->type === 'group' && $this->name) {
            return $this->name;
        }

        if ($currentUserId) {
            $otherParticipant = $this->otherParticipants($currentUserId)->first();
            if ($otherParticipant) {
                return $otherParticipant->first_name . ' ' . $otherParticipant->last_name;
            }
        }

        return 'Chat';
    }
}
