<?php

namespace App\Http\Controllers\Admin;

use App\Announcement;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\NotificationAchive;
use Auth;
use Illuminate\Support\Facades\Storage;

class AnnouncementController extends Controller
{
    // Display a listing of announcements
    public function index()
    {
        $announcements = Announcement::all();
        return view('admin.announcements.index', compact('announcements'));
    }

    public function allShow()
    {
        $announcements = Announcement::where('status', 'published')->latest()->get();
        return view('affiliate.announcements.index', compact('announcements'));
    }

    // Show the form for creating a new announcement
    public function create()
    {
        return view('admin.announcements.create');
    }

    // Store a newly created announcement in storage
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'photo' => 'required|image|mimes:jpg,png,jpeg,gif|max:2048',
            'description' => 'required|string',
            'status' => 'required|in:published,draft',
        ]);

        $announcement = new Announcement();
        $announcement->title = $request->input('title');
        $announcement->description = $request->input('description');
        $announcement->status = $request->input('status');

        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('public/announcements');
            $announcement->photo = $photoPath;
        }

        $announcement->save();

        return redirect()->route('admin.announcement.index')->with('success', 'Announcement created successfully.');
    }

    // Show the form for editing the specified announcement
    public function edit($id)
    {
        $announcement = Announcement::findOrFail($id);
        return view('admin.announcements.edit', compact('announcement'));
    }

    public function show($id)
    {
        $announcement = Announcement::where('id', $id)->where('status', 'published')->first();
        if (!$announcement) {
            abort(404);
            return;
        }
        $check = NotificationAchive::where('user_id', Auth::user()->id)->first();

        if ($check) {
            // Split existing notification_ids into an array
            $existingIds = explode(',', $check->notification_id);

            // Check if the new notification_id is already in the array
            if (!in_array($id, $existingIds)) {
                // Append the new notification_id
                $existingIds[] = $id;

                // Update the record with the new notification_id list
                $check->notification_id = implode(',', $existingIds);
                $check->save();
            }
        } else {
            // Create a new record if none exists for the user
            NotificationAchive::create([
                'user_id' => Auth::user()->id,
                'notification_id' => $id,
            ]);
        }

        return view('affiliate.announcements.view', compact('announcement'));
    }

    // Update the specified announcement in storage
    public function update(Request $request, $id)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'photo' => 'nullable|image|mimes:jpg,png,jpeg,gif|max:2048',
            'description' => 'required|string',
            'status' => 'required|in:published,draft',
        ]);

        $announcement = Announcement::findOrFail($id);
        $announcement->title = $request->input('title');
        $announcement->description = $request->input('description');
        $announcement->status = $request->input('status');

        if ($request->hasFile('photo')) {
            // Delete the old photo if it exists
            if ($announcement->photo) {
                Storage::delete($announcement->photo);
            }

            $photoPath = $request->file('photo')->store('public/announcements');
            $announcement->photo = $photoPath;
        }

        $announcement->save();

        return redirect()->route('admin.announcement.index')->with('success', 'Announcement updated successfully.');
    }

    // Remove the specified announcement from storage
    public function delete($id)
    {
        $announcement = Announcement::findOrFail($id);

        // Delete the photo if it exists
        if ($announcement->photo) {
            Storage::delete($announcement->photo);
        }

        $announcement->delete();

        return redirect()->route('admin.announcement.index')->with('success', 'Announcement deleted successfully.');
    }
}
