<?php

namespace App\Http\Controllers\Admin;

use App\User;
use App\Offer;
use App\Report;
use App\Country;
use Carbon\Carbon;
use App\OfferApplication;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        // Handle date range for main chart - default to last 7 days
        if (isset($request->date_filter)) {
            $parts = explode(' - ' , $request->date_filter);
            $date_from = $parts[0];
            $date_to = $parts[1];
        } else {
            // Default to last 7 days instead of weekly range
            $carbon_date_from = Carbon::now()->subDays(6)->startOfDay();
            $date_from = $carbon_date_from->toDateString();
            $carbon_date_to = Carbon::now()->endOfDay();
            $date_to = $carbon_date_to->toDateString();
        }
        $period = \Carbon\CarbonPeriod::create($date_from, $date_to);

        $reports = [];

        foreach ($period as $date) {

            $click = Report::where('date', $date->toDateString())
                ->get()
                ->sum('click');

            $unique = Report::where('date', $date->toDateString())
                ->get()
                ->sum('unique');

            $conversion = Report::where('date', $date->toDateString())
                ->where('status','approved')
                ->get()
                ->count();
            $revenue = Report::where('date', $date->toDateString())
                ->where('status','approved')
                ->get()
                ->sum('revenue');

            $payout = Report::where('date', $date->toDateString())
                ->where('status','approved')
                ->whereNotNull('payout')
                ->where('payout', '>', 0)
                ->get()
                ->sum('payout');

            $profit = Report::where('date', $date->toDateString())
                ->where('status','approved')
                ->get()
                ->sum('profit');

            $reports[$date->toDateString()] = compact('click','unique','conversion','revenue','payout','profit');
            
        }

        // Get time-based stats for mini charts (adapts to selected time frame)
        $timeBasedStats = [];
        $periodLength = Carbon::parse($date_from)->diffInDays(Carbon::parse($date_to)) + 1;
        
        // Determine time intervals based on period length
        if ($periodLength <= 1) {
            // For single day or less, use hourly intervals
            $interval = 'hour';
            $intervals = 24;
            $startTime = Carbon::parse($date_from)->startOfDay();
            
            for ($i = 0; $i < $intervals; $i++) {
                $timeKey = $startTime->copy()->addHours($i)->format('H:i');
                $timeStart = $startTime->copy()->addHours($i);
                $timeEnd = $timeStart->copy()->addHour();
                
                $timeClick = Report::whereBetween('created_at', [$timeStart, $timeEnd])
                    ->sum('click');
                
                $timeUnique = Report::whereBetween('created_at', [$timeStart, $timeEnd])
                    ->sum('unique');
                
                $timeConversion = Report::whereBetween('created_at', [$timeStart, $timeEnd])
                    ->where('status','approved')
                    ->count();
                
                $timeRevenue = Report::whereBetween('updated_at', [$timeStart, $timeEnd])
                    ->where('status','approved')
                    ->sum('revenue');
                
                $timePayout = Report::whereBetween('updated_at', [$timeStart, $timeEnd])
                    ->where('status','approved')
                    ->whereNotNull('payout')
                    ->where('payout', '>', 0)
                    ->sum('payout');
                
                $timeBasedStats[$timeKey] = [
                    'timeClick' => $timeClick ?: 0,
                    'timeUnique' => $timeUnique ?: 0,
                    'timeConversion' => $timeConversion ?: 0,
                    'timeRevenue' => $timeRevenue ?: 0,
                    'timePayout' => $timePayout ?: 0,
                ];
            }
        } elseif ($periodLength <= 7) {
            // For up to 7 days, use daily intervals
            $interval = 'day';
            $startTime = Carbon::parse($date_from)->startOfDay();
            
            for ($i = 0; $i < $periodLength; $i++) {
                $timeKey = $startTime->copy()->addDays($i)->format('M d');
                $timeStart = $startTime->copy()->addDays($i);
                $timeEnd = $timeStart->copy()->endOfDay();
                
                $timeClick = Report::whereBetween('date', [$timeStart->toDateString(), $timeEnd->toDateString()])
                    ->sum('click');
                
                $timeUnique = Report::whereBetween('date', [$timeStart->toDateString(), $timeEnd->toDateString()])
                    ->sum('unique');
                
                $timeConversion = Report::whereBetween('date', [$timeStart->toDateString(), $timeEnd->toDateString()])
                    ->where('status','approved')
                    ->count();
                
                $timeRevenue = Report::whereBetween('date', [$timeStart->toDateString(), $timeEnd->toDateString()])
                    ->where('status','approved')
                    ->sum('revenue');
                
                $timePayout = Report::whereBetween('date', [$timeStart->toDateString(), $timeEnd->toDateString()])
                    ->where('status','approved')
                    ->whereNotNull('payout')
                    ->where('payout', '>', 0)
                    ->sum('payout');
                
                $timeBasedStats[$timeKey] = [
                    'timeClick' => $timeClick ?: 0,
                    'timeUnique' => $timeUnique ?: 0,
                    'timeConversion' => $timeConversion ?: 0,
                    'timeRevenue' => $timeRevenue ?: 0,
                    'timePayout' => $timePayout ?: 0,
                ];
            }
        } elseif ($periodLength <= 30) {
            // For up to 30 days, use daily intervals (sample every 2nd day for performance)
            $interval = 'day';
            $startTime = Carbon::parse($date_from)->startOfDay();
            $sampleRate = max(1, floor($periodLength / 15)); // Sample up to 15 points
            
            for ($i = 0; $i < $periodLength; $i += $sampleRate) {
                $timeKey = $startTime->copy()->addDays($i)->format('M d');
                $timeStart = $startTime->copy()->addDays($i);
                $timeEnd = $timeStart->copy()->endOfDay();
                
                $timeClick = Report::whereBetween('date', [$timeStart->toDateString(), $timeEnd->toDateString()])
                    ->sum('click');
                
                $timeUnique = Report::whereBetween('date', [$timeStart->toDateString(), $timeEnd->toDateString()])
                    ->sum('unique');
                
                $timeConversion = Report::whereBetween('date', [$timeStart->toDateString(), $timeEnd->toDateString()])
                    ->where('status','approved')
                    ->count();
                
                $timeRevenue = Report::whereBetween('date', [$timeStart->toDateString(), $timeEnd->toDateString()])
                    ->where('status','approved')
                    ->sum('revenue');
                
                $timePayout = Report::whereBetween('date', [$timeStart->toDateString(), $timeEnd->toDateString()])
                    ->where('status','approved')
                    ->whereNotNull('payout')
                    ->where('payout', '>', 0)
                    ->sum('payout');
                
                $timeBasedStats[$timeKey] = [
                    'timeClick' => $timeClick ?: 0,
                    'timeUnique' => $timeUnique ?: 0,
                    'timeConversion' => $timeConversion ?: 0,
                    'timeRevenue' => $timeRevenue ?: 0,
                    'timePayout' => $timePayout ?: 0,
                ];
            }
        } else {
            // For longer periods, use weekly intervals
            $interval = 'week';
            $startTime = Carbon::parse($date_from)->startOfWeek();
            $weeks = ceil($periodLength / 7);
            
            for ($i = 0; $i < $weeks; $i++) {
                $timeKey = $startTime->copy()->addWeeks($i)->format('M d');
                $timeStart = $startTime->copy()->addWeeks($i);
                $timeEnd = $timeStart->copy()->endOfWeek();
                
                $timeClick = Report::whereBetween('date', [$timeStart->toDateString(), $timeEnd->toDateString()])
                    ->sum('click');
                
                $timeUnique = Report::whereBetween('date', [$timeStart->toDateString(), $timeEnd->toDateString()])
                    ->sum('unique');
                
                $timeConversion = Report::whereBetween('date', [$timeStart->toDateString(), $timeEnd->toDateString()])
                    ->where('status','approved')
                    ->count();
                
                $timeRevenue = Report::whereBetween('date', [$timeStart->toDateString(), $timeEnd->toDateString()])
                    ->where('status','approved')
                    ->sum('revenue');
                
                $timePayout = Report::whereBetween('date', [$timeStart->toDateString(), $timeEnd->toDateString()])
                    ->where('status','approved')
                    ->whereNotNull('payout')
                    ->where('payout', '>', 0)
                    ->sum('payout');
                
                $timeBasedStats[$timeKey] = [
                    'timeClick' => $timeClick ?: 0,
                    'timeUnique' => $timeUnique ?: 0,
                    'timeConversion' => $timeConversion ?: 0,
                    'timeRevenue' => $timeRevenue ?: 0,
                    'timePayout' => $timePayout ?: 0,
                ];
            }
        }
        
        
        if (isset($request->date_filter)) {
            $parts = explode(' - ' , $request->date_filter);
            $date_from_t = $parts[0];
            $date_to_t = $parts[1];
        } else {
            $carbon_date_from = Carbon::now();
            $date_from_t = $carbon_date_from->toDateString();
            $carbon_date_to = Carbon::now();
            $date_to_t = $carbon_date_to->toDateString();
        }
        
        $totalClick = Report::where('date','>=',$date_from_t)
            ->where('date','<=',$date_to_t)
            ->sum('click');
        $totalUnique = Report::where('date','>=',$date_from_t)
            ->where('date','<=',$date_to_t)
            ->sum('unique');
        $totalConversion = Report::where('date','>=',$date_from_t)
            ->where('date','<=',$date_to_t)
            ->where('status','approved')
            ->count();
        $totalRevenue = Report::where('date','>=',$date_from_t)
            ->where('date','<=',$date_to_t)
            ->where('status','approved')
            ->sum('revenue');
        $totalPayout = Report::where('date','>=',$date_from_t)
            ->where('date','<=',$date_to_t)
            ->where('status','approved')
            ->whereNotNull('payout')
            ->where('payout', '>', 0)
            ->sum('payout');
        $totalProfit = Report::where('date','>=',$date_from_t)
            ->where('date','<=',$date_to_t)
            ->where('status','approved')
            ->sum('profit');
        // Top Tasks
         $top_offers =  Offer::withCount([
            'reports' => function ($query) use ($date_from_t, $date_to_t) {
                $query->where('date','>=',$date_from_t)->where('date','<=',$date_to_t);
                }
            ])
            ->with(['reports' => function($query) use ($date_from_t, $date_to_t) {
                $query->where('date','>=',$date_from_t)->where('date','<=',$date_to_t)->where('status','approved');
            }])
            ->take(10)
            ->get();
        $top_countries = Country::withCount([
            'reports' => function ($query) use ($date_from_t, $date_to_t) {
                $query->where('date','>=',$date_from_t)->where('date','<=',$date_to_t);
                }
            ])
            ->with(['reports' => function($query) use ($date_from_t, $date_to_t) {
                $query->where('date','>=',$date_from_t)->where('date','<=',$date_to_t)->where('status','approved');
            }])
            ->orderByDesc("reports_count")
            ->take(10)
            ->get();
        $top_users =  User::withCount([
            'reports' => function ($query) use ($date_from_t, $date_to_t) {
                $query->where('date','>=',$date_from_t)->where('date','<=',$date_to_t);
                }
            ])
            ->with(['reports' => function($query) use ($date_from_t, $date_to_t) {
                $query->where('date','>=',$date_from_t)->where('date','<=',$date_to_t)->where('status','approved');
            }])
            ->orderByDesc("reports_count")
            ->take(10)
            ->get();
        // Last Month Data - Calculate for the same period length as current selection
        $last_month = [];
        $periodLength = Carbon::parse($date_from)->diffInDays(Carbon::parse($date_to)) + 1;
        $lastPeriodStart = Carbon::parse($date_from)->subDays($periodLength - 1)->toDateString();
        $lastPeriodEnd = Carbon::parse($date_from)->subDay()->toDateString();
        
        $last_month_click = Report::whereBetween('date',[$lastPeriodStart,$lastPeriodEnd])->sum('click');
        $last_month_unique = Report::whereBetween('date',[$lastPeriodStart,$lastPeriodEnd])->sum('unique');
        $last_month_conversion = Report::whereBetween('date',[$lastPeriodStart,$lastPeriodEnd])->where('status','approved')->count();
        $last_month_revenue = Report::whereBetween('date',[$lastPeriodStart,$lastPeriodEnd])->where('status','approved')->sum('revenue');
        $last_month_payout = Report::whereBetween('date',[$lastPeriodStart,$lastPeriodEnd])->where('status','approved')->whereNotNull('payout')->where('payout', '>', 0)->sum('payout');
        $last_month_profit = Report::whereBetween('date',[$lastPeriodStart,$lastPeriodEnd])->where('status','approved')->sum('profit');
        $last_month = compact('last_month_click','last_month_unique','last_month_conversion','last_month_revenue','last_month_payout','last_month_profit');

        $pending_affiliates = User::where('status', 'pending')->where('role', 'affiliate')->take(10)->get();
        $offer_applications = OfferApplication::where('status','false')->take(10)->get();
        
        // Prepare chart data for Analytics Overview
        $chartData = [];
        foreach ($period as $date) {
            $dateStr = $date->toDateString();
            $dayData = $reports[$dateStr] ?? [];
            $chartData[] = [
                'date' => $dateStr,
                'clicks' => $dayData['click'] ?? 0,
                'conversions' => $dayData['conversion'] ?? 0,
                'payout' => $dayData['payout'] ?? 0,
            ];
        }
        
        
        return view('admin.dashboard',compact('reports','timeBasedStats','totalClick','totalUnique','totalConversion','totalRevenue','totalPayout','totalProfit','top_offers','top_countries','top_users','pending_affiliates','offer_applications','last_month','chartData'));
    }

}
