<?php

namespace App\Http\Controllers\Admin;

use App\Finance;
use App\Http\Controllers\Controller;
use App\Invoice;
use App\Notifications\GenerateInvoice;
use App\Notifications\InvoicePaid;
use App\PaymentMethod;
use App\Report;
use App\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Notification;
use PhpParser\Node\Stmt\Switch_;

class InvoiceController extends Controller
{
    public function index()
    {
        $invoices = Invoice::orderBy('id', 'desc')->get();
        return view('admin.invoice.index', compact('invoices'));
    }

    public function generate()
    {
        $users = User::where('status', 'active')->get();
        return view('admin.invoice.generate', compact('users'));
    }

    public function getWallet(Request $request)
    {
        $payment_method = PaymentMethod::where('user_id', $request->user_id)->first();
        if ($payment_method) {
            $finance = Finance::find($payment_method->finance_id);
            return response()->json([
                'status' => 'success',
                'wallet' => $payment_method->wallet,
                'method_name' => $finance->name,
            ]);
        } else {
            return response()->json([
                'status' => 'error',
                'message' => 'No wallet found',
            ]);
        }
    }

    public function getReport(Request $request)
    {
        if (isset($request->date_filter)) {
            $parts = explode(' - ', $request->date_filter);
            $date_from = $parts[0];
            $date_to = $parts[1];
        }
        $user_id = $request->get('user_id');
        $amount = Report::where('user_id', $user_id)->where('status', 'approved')->whereBetween('date', [$date_from, $date_to])->sum('payout');
        $conversion = Report::where('user_id', $user_id)->where('status', 'approved')->whereBetween('date', [$date_from, $date_to])->count();
        return response()->json([
            'status' => 'success',
            'amount' => $amount,
            'conversion' => $conversion,
        ]);
    }


    public function autoInvoice()
    {
        if(setting('auto_invoice') != 'on'){
            return;
        }
        $minimum_invoice = setting('minimum_invoice');
        $invoice_schedule = setting('invoice_schedule');
    
        $today = Carbon::now();
        $date_to = $today->toDateString();
    
        // Calculate the start date based on the invoice schedule
        switch ($invoice_schedule) {
            case 'net7':
                $date_from = $today->subDays(7)->toDateString();
                break;
            case 'net15':
                $date_from = $today->subDays(15)->toDateString();
                break;
            case 'net30':
                $date_from = $today->subDays(30)->toDateString();
                break;
            default:
                $date_from = '1970-01-01';
                break;
        }
    
        // Fetch all unprocessed reports up to today
        $reports = Report::with('user')
            ->where('processed', false)
            ->whereDate('created_at', '<=', $date_to)
            ->get();
    
        // Group reports by user
        $reportsByUser = $reports->groupBy('user_id');
    
        foreach ($reportsByUser as $userId => $userReports) {
            $user = $userReports->first()->user;
            
            if (!$user) {
                continue; 
            }
    
            $totalAmount = $userReports->sum('payout');
            $totalConversion = $userReports->sum('conversion');
    
            // Fetch the user's payment method
            $payment_method = PaymentMethod::with('finance')->where('user_id', $user->id)->first();
    
            if (!$payment_method || !$payment_method->finance) {
                continue; // Skip if payment method or finance details are missing
            }

            // dd($payment_method,$payment_method->finance);
    
            // Ensure the total amount meets the minimum invoice requirement
            if ($totalAmount >= $minimum_invoice && $user->balance >= $totalAmount) {
                $invoice = Invoice::create([
                    'finance_name' => $payment_method->finance->name,
                    'user_id' => $user->id,
                    'conversion' => $totalConversion,
                    'amount' => $totalAmount,
                    'wallet' => $payment_method->wallet,
                    'start_date' => $date_from,
                    'end_date' => $date_to,
                ]);
    
                // Decrement user balance by the total amount
                $user->decrement('balance', $totalAmount);
    
                // Send notification
                Notification::send($user, new GenerateInvoice($invoice));
    
                // Mark all processed reports
                foreach ($userReports as $report) {
                    $report->update(['processed' => true]);
                }
            }
        }
    
        return response()->json(['success' => 'Reports generated successfully']);
    }
    

    public function store(Request $request)
    {
        $this->validate($request, [
            'user_id' => 'required',
            'finance_name' => 'required|string|max:255',
            'wallet' => 'required|string|max:255',
            'conversion' => 'required|max:255',
            'amount' => 'required|max:255',
        ]);

        if (isset($request->date_filter)) {
            $parts = explode(' - ', $request->date_filter);
            $date_from = $parts[0];
            $date_to = $parts[1];
        } else {
            // If no date filter is provided, set default date range to today
            $date_from = Carbon::now()->startOfDay()->toDateString();
            $date_to = Carbon::now()->endOfDay()->toDateString();
        }

        $user = User::find($request->user_id);
        if ($user->balance < $request->amount) {
            return back()->with('error', 'Insufficient balance');
        }

        // Create the invoice
        $invoice = Invoice::create([
            'finance_name' => $request->finance_name,
            'user_id' => $request->user_id,
            'conversion' => $request->conversion,
            'amount' => $request->amount,
            'wallet' => $request->wallet,
            'start_date' => $date_from,
            'end_date' => $date_to,
        ]);

        // Decrement user's balance by the invoice amount
        $user->decrement('balance', $request->amount);

        // Send notification
        Notification::send($user, new GenerateInvoice($invoice));

        // Mark reports as processed within the date range for the specified user
        Report::where('user_id', $request->user_id)
            ->whereBetween('created_at', [$date_from, $date_to])
            ->update(['processed' => true]);

        return redirect()->route('admin.invoice.index')->with('success', 'Invoice created successfully');
    }


    public function paid($id)
    {
        $invoice = Invoice::find($id);
        $invoice->update([
            'status' => true,
            'paid_at' => Carbon::now(),
        ]);
        $user = User::find($invoice->user_id);
        Notification::send($user, new InvoicePaid($invoice));
        return redirect()->route('admin.invoice.index')->with('success', 'Invoice paid successfully');
    }

    // public function managerInvoice()
    // {
    //     $managers = User::where('role','manager')->with('invoices')->get();
    //     return $managers;
    //     $invoices = Invoice::whereIn('user_id',$managers->id)->orderBy('id', 'desc')->get();
    //     return $invoices;
    //     return view('admin.managerinvoice.index',compact('managers'));
    // }
    // public function managerInvoicePaid($id)
    // {

    // }
}
