<?php

namespace App\Http\Controllers\Affiliate;

use App\Country;
use App\Finance;
use App\Http\Controllers\Controller;
use App\Http\Requests\AffiliateProfileUpdate;
use App\PaymentMethod;
use App\Postback;
use App\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Laravel\Ui\Presets\React;

class AccountController extends Controller
{
    public function index()
    {
        $countries = Country::all();
        $finances = Finance::all();
        $refer_list = User::where('refer_id', Auth::user()->id)->where('role', 'affiliate')->get();
        return view('affiliate.account.index', compact('countries', 'finances', 'refer_list'));
    }

    public function postbackurl()
    {
        $countries = Country::all();
        $finances = Finance::all();
        $refer_list = User::where('refer_id', Auth::user()->id)->where('role', 'affiliate')->get();
        return view('affiliate.account.postback', compact('countries', 'finances', 'refer_list'));
    }

    public function update(AffiliateProfileUpdate $request)
    {
        $user = Auth::user();

        // Update profile data
        $user->update([
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'email' => $request->email,
            'country_id' => $request->country_id,
            'skype' => $request->skype,
        ]);

        // Add avatar if provided
        try {
            // Check if a new avatar file is provided
            if ($request->hasFile('avatar')) {
                // Clear existing avatar
                $user->clearMediaCollection('avatar');
                // Add new avatar
                $user->addMedia($request->file('avatar'))->toMediaCollection('avatar');
            }
        } catch (\Exception $e) {
            return back()->with('error', 'An error occurred while uploading the avatar.');
        }

        // Change password if requested
        if ($request->filled('password')) {
            $this->validate($request, [
                'password' => 'required|min:6|confirmed',
                'old_password' => 'required',
            ]);

            if (\Hash::check($request->old_password, $user->password)) {
                $user->update([
                    'password' => \Hash::make($request->password),
                ]);
                return back()->with('success', 'Password changed successfully');
            } else {
                return back()->with('error', 'Old password is incorrect');
            }
        }

        return back()->with('success', 'Profile updated successfully');
    }


public function paymentUpdate(Request $request)
{
    // Find the selected finance method by ID
    $finance = Finance::find($request->finance_id);

    if (!$finance) {
        return back()->withErrors(['finance_id' => "Invalid finance method selected."]);
    }

    // Conditional validation based on the finance method's name
    $this->validate($request, [
        'finance_id' => 'required',
        'wallet' => 'sometimes|required',
        'bank_name' => $finance->name === 'Bank' ? 'required' : 'nullable',
        'routing_number' => $finance->name === 'Bank' ? 'required' : 'nullable',
        'swift_code' => $finance->name === 'Bank' ? 'required' : 'nullable',
        'account_number' => $finance->name === 'Bank' ? 'required' : 'nullable',
        'beneficiary_name' => $finance->name === 'Bank' ? 'required' : 'nullable',
    ]);

    // Prepare wallet data
    $wallet = $request->wallet;

    if ($finance->name === 'Bank') {
        $wallet = json_encode([
            'bank_name' => $request->bank_name,
            'routing_number' => $request->routing_number,
            'swift_code' => $request->swift_code,
            'account_number' => $request->account_number,
            'beneficiary_name' => $request->beneficiary_name,
        ]);
    }

    // Find or create the payment method for the user
    $paymentMethod = PaymentMethod::firstOrNew(['user_id' => Auth::user()->id]);
    $paymentMethod->finance_id = $request->finance_id;
    $paymentMethod->wallet = $wallet;
    $paymentMethod->save();

    return back()->with('payment_info_success', 'Payment method updated successfully');
}

    

    public function postback(Request $request)
    {
        if ($request->url) {
            $this->validate($request, [
                'url' => 'required|string',
            ]);
            $postback = Postback::where('user_id', Auth::user()->id)->first();
            if ($postback) {
                $postback->update([
                    'url' => $request->url,
                ]);
            } else {
                Postback::create([
                    'user_id' => Auth::user()->id,
                    'url' => $request->url,
                ]);
            }
            return back()->with('postback', 'Postback url update successfully');
        } else {
            $postback = Postback::where('user_id', Auth::user()->id)->first();
            if ($postback) {
                $postback->delete();
            }
        }
        return back()->with('postback', 'Postback url update successfully');
    }
}
