<?php

namespace App\Http\Controllers\Affiliate;

use App\Country;
use App\Http\Controllers\Controller;
use App\Invoice;
use App\Offer;
use App\Report;
use App\TopOffer;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;

class DashboardController extends Controller
{

    public function index(Request $request)
    {
       
        if ($request->filled('start') && $request->filled('end')) {
            try {
                $date_from = Carbon::parse($request->input('start'))->startOfDay();
                $date_to = Carbon::parse($request->input('end'))->endOfDay();
            } catch (\Throwable $e) {
                $date_from = Carbon::now()->startOfDay();
                $date_to = Carbon::now()->endOfDay();
            }
        } elseif (isset($request->date_filter)) {
            switch ($request->date_filter) {
                case 'today':
                    $date_from = Carbon::now()->startOfDay();
                    $date_to = Carbon::now()->endOfDay();
                    break;
                case '7D':
                    $date_from = Carbon::now()->subDays(7)->startOfDay();
                    $date_to = Carbon::now()->endOfDay();
                    break;
                case '14D':
                    $date_from = Carbon::now()->subDays(14)->startOfDay();
                    $date_to = Carbon::now()->endOfDay();
                    break;
                case '30D':
                    $date_from = Carbon::now()->subDays(30)->startOfDay();
                    $date_to = Carbon::now()->endOfDay();
                    break;
                case '90D':
                    $date_from = Carbon::now()->subDays(90)->startOfDay();
                    $date_to = Carbon::now()->endOfDay();
                    break;
                default:
                    $date_from = Carbon::now()->startOfDay();
                    $date_to = Carbon::now()->endOfDay();
                    break;
            }
        } else {
            // Default to last 7 days instead of today only
            $date_from = Carbon::now()->subDays(6)->startOfDay();
            $date_to = Carbon::now()->endOfDay();
        }

        $reports = [];

        $period = \Carbon\CarbonPeriod::create($date_from, $date_to);
        $liveConversion = Report::where('created_at', '>=', Carbon::now()->subMinutes(30))
            ->where('user_id', Auth::id())
            ->where('status', 'approved')
            ->count();

        // Get hourly stats for stats cards (today's hourly data)
        $hourlyStats = [];
        $today = Carbon::now()->startOfDay();
        
        // Get today's hourly data using created_at
        for ($i = 0; $i < 24; $i++) {
            $hour = sprintf('%02d:00', $i);
            $hourStart = $today->copy()->addHours($i);
            $hourEnd = $hourStart->copy()->addHour();
            
            $hourlyClick = Report::whereBetween('created_at', [$hourStart, $hourEnd])
                ->where('user_id', Auth::id())
                ->sum('click');
            
            $hourlyUnique = Report::whereBetween('created_at', [$hourStart, $hourEnd])
                ->where('user_id', Auth::id())
                ->sum('unique');
            
            $hourlyConversion = Report::whereBetween('updated_at', [$hourStart, $hourEnd])
                ->where('status','approved')
                ->where('user_id', Auth::id())
                ->count();
            
            $hourlyPayout = Report::whereBetween('updated_at', [$hourStart, $hourEnd])
                ->where('status','approved')
                ->where('user_id', Auth::id())
                ->whereNotNull('payout')
                ->where('payout', '>', 0)
                ->sum('payout');
            
            $hourlyStats[$hour] = compact('hourlyClick','hourlyUnique','hourlyConversion','hourlyPayout');
        }

        foreach ($period as $date) {
            $click = Report::where('date', $date->toDateString())
                ->where('user_id', Auth::id())
                ->sum('click');

            $unique = Report::where('date', $date->toDateString())
                ->where('user_id', Auth::id())
                ->sum('unique');

            $conversion = Report::where('date', $date->toDateString())
                ->where('user_id', Auth::id())
                ->where('status', 'approved')
                ->count();

            $payout = Report::where('date', $date->toDateString())
                ->where('user_id', Auth::id())
                ->where('status', 'approved')
                ->whereNotNull('payout')
                ->where('payout', '>', 0)
                ->sum('payout');

            $reports[$date->toDateString()] = compact('click', 'unique', 'conversion', 'payout', 'liveConversion');
        }

        // Total data calculations
        $totalClick = Report::whereBetween('date', [$date_from->toDateString(), $date_to->toDateString()])
            ->where('user_id', Auth::id())
            ->sum('click');

        $totalUnique = Report::whereBetween('date', [$date_from->toDateString(), $date_to->toDateString()])
            ->where('user_id', Auth::id())
            ->sum('unique');

        $totalConversion = Report::whereBetween('date', [$date_from->toDateString(), $date_to->toDateString()])
            ->where('user_id', Auth::id())
            ->where('status', 'approved')
            ->count();

        $totalPayout = Report::whereBetween('date', [$date_from->toDateString(), $date_to->toDateString()])
            ->where('user_id', Auth::id())
            ->where('status', 'approved')
            ->whereNotNull('payout')
            ->where('payout', '>', 0)
            ->sum('payout');

        $top_offers = Offer::withCount([
            'reports as reports_count' => function ($query) use ($date_from, $date_to) {
                $query->whereBetween('date', [$date_from->toDateString(), $date_to->toDateString()])
                    ->where('user_id', Auth::id());
            },
        ])
            ->with(['reports' => function ($query) use ($date_from, $date_to) {
                $query->whereBetween('date', [$date_from->toDateString(), $date_to->toDateString()])
                    ->where('status', 'approved')
                    ->where('user_id', Auth::id());
            }])
            ->orderByDesc('reports_count')
            ->take(5)
            ->get();

        $top_countries = Country::withCount([
            'reports as reports_count' => function ($query) use ($date_from, $date_to) {
                $query->where('user_id', Auth::id())
                    ->whereBetween('date', [$date_from->toDateString(), $date_to->toDateString()]);
            },
        ])
            ->with(['reports' => function ($query) use ($date_from, $date_to) {
                $query->whereBetween('date', [$date_from->toDateString(), $date_to->toDateString()])
                    ->where('status', 'approved')
                    ->where('user_id', Auth::id());
            }])
            ->orderByDesc('reports_count')
            ->take(5)
            ->get();

        $last_month = [];
        $startMonth = $date_from->copy()->subMonth()->startOfMonth()->toDateString();
        $endMonth = $date_to->copy()->subMonth()->endOfMonth()->toDateString();

        $last_month_click = Report::whereBetween('date', [$startMonth, $endMonth])
            ->where('user_id', Auth::id())
            ->where('status', 'click')
            ->sum('click');

        $last_month_unique = Report::whereBetween('date', [$startMonth, $endMonth])
            ->where('user_id', Auth::id())
            ->where('status', 'click')
            ->sum('unique');

        $last_month_conversion = Report::whereBetween('date', [$startMonth, $endMonth])
            ->where('user_id', Auth::id())
            ->where('status', 'approved')
            ->count();

        $last_month_payout = Report::whereBetween('date', [$startMonth, $endMonth])
            ->where('user_id', Auth::id())
            ->where('status', 'approved')
            ->whereNotNull('payout')
            ->where('payout', '>', 0)
            ->sum('payout');

        $countryData = Report::select('country_id', \DB::raw('count(*) as total'))
            ->where('user_id', Auth::id())
            ->groupBy('country_id')
            ->orderBy('total', 'desc')
            ->limit(4)
            ->with('country')
            ->get();

        $osData = Report::select('os_name', \DB::raw('count(*) as total'))
            ->where('user_id', Auth::id())
            ->groupBy('os_name')
            ->orderBy('total', 'desc')
            ->limit(4)
            ->get();

        $browserData = Report::select('browser', \DB::raw('count(*) as total'))
            ->where('user_id', Auth::id())
            ->groupBy('browser')
            ->orderBy('total', 'desc')
            ->limit(4)
            ->get();

        // Prepare data for the chart
        $countryChartData = [
            'series' => $countryData->pluck('total')->toArray(),
            'labels' => $countryData->pluck('country.name')->toArray(),
        ];

        $osChartData = [
            'series' => $osData->pluck('total')->toArray(),
            'labels' => $osData->pluck('os_name')->toArray(),
        ];

        $browserChartData = [
            'series' => $browserData->pluck('total')->toArray(),
            'labels' => $browserData->pluck('browser')->toArray(),
        ];

        $last_month = compact('last_month_click', 'last_month_unique', 'last_month_conversion', 'last_month_payout');

        // Chart data grouped by reports.date within the selected range
        $data = Report::select(
                DB::raw('date as date'),
                DB::raw('SUM(click) as clicks'),
                DB::raw('SUM(CASE WHEN status = "approved" THEN 1 ELSE 0 END) as conversions'),
                DB::raw('SUM(CASE WHEN status = "approved" AND payout IS NOT NULL AND payout > 0 THEN payout ELSE 0 END) as payout')
            )
            ->where('user_id', Auth::id())
            ->whereBetween('date', [$date_from->toDateString(), $date_to->toDateString()])
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Build contiguous daily series for the selected range
        $byDate = $data->keyBy('date');
        $series = [];
        foreach (\Carbon\CarbonPeriod::create($date_from->copy()->startOfDay(), $date_to->copy()->startOfDay()) as $day) {
            $d = $day->toDateString();
            $row = $byDate->get($d);
            $series[] = [
                'date' => $d,
                'clicks' => $row ? (int)$row->clicks : 0,
                'conversions' => $row ? (int)$row->conversions : 0,
                'payout' => $row ? (float)$row->payout : 0.0,
            ];
        }

        $chartData = $series;

        $payoutData = Invoice::where('user_id', Auth::id())->get();

        //$totalPayout = $payoutData->sum('amount');
        $lastMonthPayout = $payoutData->where('created_at', '>=', Carbon::now()->subMonth())->sum('amount');
        $lastYearPayout = $payoutData->where('created_at', '>=', Carbon::now()->subYear())->sum('amount');

        $payoutDataArray = [
            'total' => $totalPayout,
            'last_month' => $lastMonthPayout,
            'last_year' => $lastYearPayout,
        ];

        $conversionData = Report::where('user_id', Auth::id())
            ->where('status', 'approved')
            ->get();

        $totalConversionT = $conversionData->count();
        $lastMonthConversionT = $conversionData->where('created_at', '>=', Carbon::now()->subMonth())->count();
        $lastYearConversionT = $conversionData->where('created_at', '>=', Carbon::now()->subYear())->count();

        $conversionDataArray = [
            'total' => $totalConversionT,
            'last_month' => $lastMonthConversionT,
            'last_year' => $lastYearConversionT,
        ];

        $clickData = Report::where('user_id', Auth::id())
            ->get();

        $totalClickT = $clickData->sum('click');
        $lastMonthClickT = $clickData->where('created_at', '>=', Carbon::now()->subMonth())->sum('click');
        $lastYearClickT = $clickData->where('created_at', '>=', Carbon::now()->subYear())->sum('click');

        $clickDataArray = [
            'total' => $totalClickT,
            'last_month' => $lastMonthClickT,
            'last_year' => $lastYearClickT,
        ];

        // Get top offers from the global top offers table
        $globalTopOffers = TopOffer::getActiveTopOffers(5);

        return view('affiliate.dashboard', compact('reports', 'hourlyStats', 'totalClick', 'totalUnique', 'totalConversion', 'totalPayout', 'top_offers', 'top_countries', 'last_month', 'request', 'liveConversion', 'countryChartData', 'browserChartData', 'osChartData', 'chartData', 'payoutDataArray', 'clickDataArray', 'conversionDataArray', 'globalTopOffers'));
    }



}
