<?php

namespace App\Http\Controllers\Affiliate;

use App\Offer;
use App\Country;
use App\Category;
use App\OfferApplication;
use App\User;
use App\Device;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Arr;
use Session;

class OfferController extends Controller
{

    public function index(Request $request)
    {
        // Initialize variables with empty defaults
        $name = '';
        $category_id = [];
        $country_id = [];
        $status = null;

        // Ensure a default view is set (default to list view)
        if (!Session::has('offer_view_type')) {
            Session::put('offer_view_type', 'glary');
        }

        // Check if filter is applied
        if ($request->filter) {
            // Assign filter values
            $status = $request->status;
            $name = $request->name;

            // Ensure category_id and country_id are arrays
            $category_id = Arr::wrap($request->category_id);
            $country_id = Arr::wrap($request->country_id);

            // Base query with eager loads
            $offers = Offer::where('is_smart', 0)
                ->with([
                    'countries',
                    'devices',
                    'category',
                    'verticle',
                    'countryPayouts.country',
                    'countryPayouts.devices.device',
                    'applications' => function ($q) {
                        $q->where('user_id', Auth::id());
                    },
                ])
                // Filter based on the 'requestApproved' status or application status
                ->when($status === 'requestApproved', function ($query) {
                    $query->where('status', 'requestApproved')
                        ->whereDoesntHave('applications', function ($subQuery) {
                            $subQuery->where('user_id', Auth::user()->id);
                        });
                })

                ->when($status === 'Approved', function ($query) {
                    $query->where('status', 'requestApproved')
                        ->whereHas('applications', function ($subQuery) {
                            $subQuery->where('user_id', Auth::user()->id)
                                ->where('status', true);
                        });
                })
                ->when($status === 'Approval Pending', function ($query) {
                    $query->where('status', 'requestApproved')
                        ->whereHas('applications', function ($subQuery) {
                            $subQuery->where('user_id', Auth::user()->id)
                                ->where('status', false);
                        });
                })
                // Filter by name or ID if not empty
                ->when(!empty($name), function ($query) use ($name) {
                    return $query->where('name', 'like', '%' . $name . '%')
                        ->orWhere('id', $name);
                })
                // Filter by category if not empty
                ->when(!empty($category_id), function ($query) use ($category_id) {
                    return $query->whereIn('category_id', $category_id);
                })
                // Filter by country if not empty
                ->when(!empty($country_id), function ($query) use ($country_id) {
                    return $query->whereHas('countries', function ($query) use ($country_id) {
                        $query->whereIn('country_id', $country_id);
                    });
                })
                ->latest()->paginate(10);
        } else {
            // Default offers query without filters, sorted by latest, with eager loads
            $offers = Offer::where('is_smart', 0)
                ->with([
                    'countries',
                    'devices',
                    'category',
                    'verticle',
                    'countryPayouts.country',
                    'countryPayouts.devices.device',
                    'applications' => function ($q) {
                        $q->where('user_id', Auth::id());
                    },
                ])
                ->latest()->paginate(20);
        }

        // Fetch all categories, countries, and devices for filtering options
        $categories = Category::all();
        $countries = Country::all();
        $devices = Device::all();

        $list = false;
        if (Session::get('offer_view_type') == 'glary') {
            $list = true;
        }
        // Return the view with the required data
        return view('affiliate.offer.index', compact('offers', 'categories', 'countries', 'devices','list'));
    }

    public function viewSwitch(){
        if(Session::get('offer_view_type')=='glary'){
            Session::put('offer_view_type','list');
        }else{
            Session::put('offer_view_type','glary');
        }
        return redirect()->back();
    }



    public function smartlink(Request $request)
    {
        $name = '';
        $category_id = [];
        $country_id = [];
        $status = ['active', 'requestApproved'];

        $offers = Offer::where('is_smart', 1)->whereIN('status', $status)->paginate(20);

        $categories = Category::all();
        $countries = Country::all();
        $devices = Device::all();
        return view('affiliate.offer.smartlink', compact('offers', 'categories', 'countries', 'devices'));
    }
    public function show($id)
    {
        $offer = Offer::find($id);
        return view('affiliate.offer.show', compact('offer'));
    }

    public function application(Request $request, $id)
    {
        $this->validate($request, [
            'message' => 'required|string|max:255'
        ]);
        $application = OfferApplication::create([
            'offer_id' => $id,
            'user_id' => Auth::id(),
            'message' => $request->message,
        ]);
        $user = Auth::user();
        $admin = User::where('role', 'admin')->first();
        $manager = $user->manager;
        Notification::send($user, new \App\Notifications\OfferApplication($application));
        Notification::send($admin, new \App\Notifications\OfferApplicationAdmin($application));
        Notification::send($manager, new \App\Notifications\OfferApplicationManager($application));
        return back()->with('success', 'Your application has been sent successfully');
    }

    public function api($id)
    {
        $offer = Offer::find($id);
        if (!$offer) {
            return '<p>No data found</p>';
        }

        return view('affiliate.offer.view', compact('offer'));
    }
}
