<?php

namespace App\Http\Controllers\Affiliate;

use App\Advertiser;
use App\Country;
use App\Http\Controllers\Controller;
use App\Offer;
use App\Report;
use App\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ReportController extends Controller
{

    public function day(Request $request)
    {
        // Default selected fields
        $selectedFields = [
            'offer' => $request->has('offer'),
            'country' => $request->has('country'),
            'device' => $request->has('device'),
            'os' => $request->has('os'),
            'browser' => $request->has('browser'),
            'click' => $request->has('click'),
            'unique' => $request->has('unique'),
            'payout' => $request->has('payout'),
            'epc' => $request->has('epc'),
            'conversions' => $request->has('conversions'),
        ];

        // Date range handling (same as before)
        if ($request->has('last_weak')) {
            $carbon_date_from = Carbon::now()->subWeek()->startOfWeek();
            $date_from = $carbon_date_from->toDateString();
            $carbon_date_to = Carbon::now()->subWeek()->endOfWeek();
            $date_to = $carbon_date_to->toDateString();
        } elseif ($request->has('this_weak')) {
            $carbon_date_from = Carbon::now()->startOfWeek();
            $date_from = $carbon_date_from->toDateString();
            $carbon_date_to = Carbon::now()->endOfWeek();
            $date_to = $carbon_date_to->toDateString();
        } elseif ($request->has('date_filter')) {
            $parts = explode(' - ', $request->date_filter);
            $date_from = $parts[0];
            $date_to = $parts[1];
        } else {
            $carbon_date_from = new Carbon('3 days ago');
            $date_from = $carbon_date_from->toDateString();
            $carbon_date_to = new Carbon('today');
            $date_to = $carbon_date_to->toDateString();
        }

        // Collect filters from the request
        $offer_id = $request->get('offer_id', []);
        $country_id = $request->get('country_id', []);
        $advertiser_id = $request->get('advertiser_id', []);

        // Build the query based on selected fields
        $query = Report::with('offer', 'country')->whereBetween('date', [$date_from, $date_to])
            ->where('user_id', Auth::id())
            ->when(!empty($offer_id), function ($query) use ($offer_id) {
                return $query->whereIn('offer_id', $offer_id);
            })
            ->when(!empty($country_id), function ($query) use ($country_id) {
                return $query->whereIn('country_id', $country_id);
            })
            ->when(!empty($advertiser_id), function ($query) use ($advertiser_id) {
                return $query->whereHas('advertiser', function ($query) use ($advertiser_id) {
                    return $query->whereIn('advertiser_id', $advertiser_id);
                });
            });

    // For debugging: capture raw matching row count before aggregation
    $rawQuery = clone $query;
    $raw_matching_count = $rawQuery->count();

    // dd(json_encode($query->get()));

    $reports = $query->groupBy('date')
    ->selectRaw(
        'date,' .
        'GROUP_CONCAT(DISTINCT countries.name) as country_name,' .
        'GROUP_CONCAT(DISTINCT os_name) as os_name,' .
        'GROUP_CONCAT(DISTINCT browser) as browser,' .
        'GROUP_CONCAT(DISTINCT device_brand) as device_brand,' .
        'GROUP_CONCAT(DISTINCT device_model) as device_model,' .
        'GROUP_CONCAT(DISTINCT user_agent) as user_agent,' .
        'GROUP_CONCAT(DISTINCT ip_address) as ip_address,' .
        'GROUP_CONCAT(DISTINCT aff_sub_1) as aff_sub_1,' .
        'GROUP_CONCAT(DISTINCT aff_sub_2) as aff_sub_2,' .
        'GROUP_CONCAT(DISTINCT aff_sub_3) as aff_sub_3,' .
        'GROUP_CONCAT(DISTINCT geo_city) as geo_city,' .
        'GROUP_CONCAT(DISTINCT geo_region) as geo_region,' .
        'SUM(click) as total_clicks,' .
        'SUM(`unique`) as total_uniques,' .
        'SUM(CASE WHEN status = "approved" THEN payout ELSE 0 END) as total_payouts,' .
        'COUNT(CASE WHEN status = "approved" THEN 1 END) as total_conversions,' .
        'AVG(CASE WHEN status = "approved" THEN payout / NULLIF(click, 0) ELSE NULL END) as epc'
    )
    ->leftJoin('countries', 'reports.country_id', '=', 'countries.id')
    ->get()
    ->keyBy('date');

    
    
        // Preparing total and selected fields for display
        $countries = Country::all();
        $users = User::where('role', 'affiliate')->get();
        $advertisers = Advertiser::all();
        $offers = Offer::all();
        $total_clicks = $reports->sum('total_clicks'); // Replace with actual calculation logic
        $total_uniques = $reports->sum('total_uniques');
        $total_conversions = $reports->sum('total_conversions');
        $total_payouts = $reports->sum('total_payouts');
        $total_epc = $reports->sum('epc');

        // Prepare totals array for returning to the view
        $total = [
            'clicks' => $total_clicks,
            'uniques' => $total_uniques,
            'conversion' => $total_conversions,
            'payouts' => $total_payouts,
            'epc' => $total_epc,
        ];

        // Debug info (only when app debug enabled)
        $debug_info = null;
        if (config('app.debug')) {
            $debug_info = [
                'raw_matching_count' => $raw_matching_count,
                'grouped_count' => is_countable($reports) ? count($reports) : ($reports instanceof \Illuminate\Support\Collection ? $reports->count() : null),
                'sample_sql' => $rawQuery->toSql(),
                'bindings' => $rawQuery->getBindings(),
                'date_from' => $date_from,
                'date_to' => $date_to,
                'user_id' => Auth::id(),
                'reports_data' => $reports->toArray(),
            ];
        }

        // Return to view with the selected fields
        return view('affiliate.report.day', compact('countries', 'users', 'offers', 'advertisers', 'reports', 'selectedFields', 'total', 'debug_info'));
    }

    public function offer(Request $request)
    {
        $offer_id = Report::query()->pluck('offer_id')->unique()->toArray();
        $offers = $offer_id;
        $country_id = [];
        if (isset($request->date_filter)) {
            $parts = explode(' - ', $request->date_filter);
            $date_from = $parts[0];
            $date_to = $parts[1];
            $offer_id = $request->get('offer_id');
            $country_id = $request->get('country_id');
        } else {
            $carbon_date_from = new Carbon('last Monday');
            $date_from = $carbon_date_from->toDateString();
            $carbon_date_to = new Carbon('this Sunday');
            $date_to = $carbon_date_to->toDateString();
        }

        $total = Report::affiliateTotal($date_from, $date_to, $offer_id, $country_id);

        // Initialize reports array
        $reports = [];

        foreach ($offers as $offer_id) {
            $query = Report::where('offer_id', $offer_id)
                ->whereBetween('date', [$date_from, $date_to])
                ->where('user_id', Auth::id())
                ->when(!empty($country_id), function ($query) use ($country_id) {
                    return $query->whereIn('country_id', $country_id);
                });
            $click = $query->sum('click');
            $unique = $query->sum('unique');
            $conversion = $query->where('status', 'approved')->count();
            $revenue = $query->where('status', 'approved')->sum('revenue');
            $payout = $query->where('status', 'approved')->sum('payout');
            $profit = $query->where('status', 'approved')->sum('profit');
            $reports[$offer_id] = compact('click', 'unique', 'conversion', 'revenue', 'payout', 'profit');
        }

        $countries = Country::all();
        $users = User::where('role', 'affiliate')->get();
        $advertisers = Advertiser::all();
        $offers = Offer::all();
        return view('affiliate.report.offer', compact('countries', 'users', 'offers', 'advertisers', 'reports', 'total'));
    }

    public function country(Request $request)
    {
        $offer_id = [];
        $country_id = Report::query()->pluck('country_id')->unique()->toArray();
        if (isset($request->date_filter)) {
            $parts = explode(' - ', $request->date_filter);
            $date_from = $parts[0];
            $date_to = $parts[1];
            $offer_id = $request->get('offer_id');
            $country_id = $request->get('country_id');
        } else {
            $carbon_date_from = new Carbon('last Monday');
            $date_from = $carbon_date_from->toDateString();
            $carbon_date_to = new Carbon('this Sunday');
            $date_to = $carbon_date_to->toDateString();
        }

        $total = Report::affiliateTotal($date_from, $date_to, $offer_id, $country_id);
        $countries = $request->country_id ?? Report::query()->distinct()->pluck('country_id');
        $reports = [];
        foreach ($countries as $country_id) {
            $query = Report::where('country_id', $country_id)
                ->whereBetween('date', [$date_from, $date_to])
                ->where('user_id', Auth::id())
                ->when(!empty($offer_id), function ($query) use ($offer_id) {
                    return $query->whereIn('offer_id', $offer_id);
                });
            $click = $query->sum('click');
            $unique = $query->sum('unique');
            $conversion = $query->where('status', 'approved')->count();
            $revenue = $query->where('status', 'approved')->sum('revenue');
            $payout = $query->where('status', 'approved')->sum('payout');
            $profit = $query->where('status', 'approved')->sum('profit');
            $reports[$country_id] = compact('click', 'unique', 'conversion', 'revenue', 'payout', 'profit');
        }
        $countries = Country::all();
        $advertisers = Advertiser::all();
        $offers = Offer::all();
        return view('affiliate.report.country', compact('countries', 'offers', 'advertisers', 'reports', 'total'));
    }

    public function aff_sub_1(Request $request)
    {
        $offer_id = [];
        $country_id = [];
        if (isset($request->date_filter)) {
            $parts = explode(' - ', $request->date_filter);
            $date_from = $parts[0];
            $date_to = $parts[1];
            $offer_id = $request->get('offer_id');
            $country_id = $request->get('country_id');
        } else {
            $carbon_date_from = new Carbon('last Monday');
            $date_from = $carbon_date_from->toDateString();
            $carbon_date_to = new Carbon('this Sunday');
            $date_to = $carbon_date_to->toDateString();
        }
        $total = Report::affiliateTotal($date_from, $date_to, $offer_id, $country_id);
        
        // Get distinct aff_sub_1 values with additional data
        $aff_sub_1_reports = Report::whereBetween('date', [$date_from, $date_to])
            ->where('user_id', Auth::id())
            ->when(!empty($offer_id), function ($query) use ($offer_id) {
                return $query->whereIn('offer_id', $offer_id);
            })
            ->when(!empty($country_id), function ($query) use ($country_id) {
                return $query->whereIn('country_id', $country_id);
            })
            ->groupBy('aff_sub_1')
            ->selectRaw(
                'aff_sub_1,' .
                'GROUP_CONCAT(DISTINCT offers.name) as offer_name,' .
                'GROUP_CONCAT(DISTINCT countries.name) as country_name,' .
                'GROUP_CONCAT(DISTINCT device_brand) as device_brand,' .
                'GROUP_CONCAT(DISTINCT device_model) as device_model,' .
                'GROUP_CONCAT(DISTINCT os_name) as os_name,' .
                'GROUP_CONCAT(DISTINCT browser) as browser,' .
                'GROUP_CONCAT(DISTINCT aff_sub_2) as aff_sub_2,' .
                'GROUP_CONCAT(DISTINCT aff_sub_3) as aff_sub_3,' .
                'SUM(click) as click,' .
                'SUM(`unique`) as unique_count,' .
                'COUNT(CASE WHEN reports.status = "approved" THEN 1 END) as conversion,' .
                'SUM(CASE WHEN reports.status = "approved" THEN reports.payout ELSE 0 END) as payout'
            )
            ->leftJoin('offers', 'reports.offer_id', '=', 'offers.id')
            ->leftJoin('countries', 'reports.country_id', '=', 'countries.id')
            ->get()
            ->keyBy('aff_sub_1');
            
        $reports = $aff_sub_1_reports->toArray();
        
        $countries = Country::all();
        $users = User::where('role', 'affiliate')->get();
        $advertisers = Advertiser::all();
        $offers = Offer::all();
        return view('affiliate.report.aff_sub_1', compact('countries', 'users', 'offers', 'advertisers', 'reports', 'total'));
    }

    public function aff_sub_2(Request $request)
    {
        $offer_id = [];
        $country_id = [];
        if (isset($request->date_filter)) {
            $parts = explode(' - ', $request->date_filter);
            $date_from = $parts[0];
            $date_to = $parts[1];
            $offer_id = $request->get('offer_id');
            $country_id = $request->get('country_id');
        } else {
            $carbon_date_from = new Carbon('last Monday');
            $date_from = $carbon_date_from->toDateString();
            $carbon_date_to = new Carbon('this Sunday');
            $date_to = $carbon_date_to->toDateString();
        }
        $total = Report::affiliateTotal($date_from, $date_to, $offer_id, $country_id);
        $aff_sub_2 = Report::query()->distinct()->pluck('aff_sub_2')->toArray();
        $reports = [];
        foreach ($aff_sub_2 as $aff_sub_2) {
            $query = Report::where('aff_sub_2', $aff_sub_2)
                ->whereBetween('date', [$date_from, $date_to])
                ->where('user_id', Auth::id())
                ->when(!empty($offer_id), function ($query) use ($offer_id) {
                    return $query->whereIn('offer_id', $offer_id);
                });
            $click = $query->sum('click');
            $unique = $query->sum('unique');
            $conversion = $query->where('status', 'approved')->count();
            $revenue = $query->where('status', 'approved')->sum('revenue');
            $payout = $query->where('status', 'approved')->sum('payout');
            $profit = $query->where('status', 'approved')->sum('profit');
            $reports[$aff_sub_2] = compact('click', 'unique', 'conversion', 'revenue', 'payout', 'profit');
        }
        $countries = Country::all();
        $users = User::where('role', 'affiliate')->get();
        $advertisers = Advertiser::all();
        $offers = Offer::all();
        return view('affiliate.report.aff_sub_2', compact('countries', 'users', 'offers', 'advertisers', 'reports', 'total'));
    }


    public function aff_sub_3(Request $request)
    {
        $offer_id = [];
        $country_id = [];
        if (isset($request->date_filter)) {
            $parts = explode(' - ', $request->date_filter);
            $date_from = $parts[0];
            $date_to = $parts[1];
            $offer_id = $request->get('offer_id');
            $country_id = $request->get('country_id');
        } else {
            $carbon_date_from = new Carbon('last Monday');
            $date_from = $carbon_date_from->toDateString();
            $carbon_date_to = new Carbon('this Sunday');
            $date_to = $carbon_date_to->toDateString();
        }
        $total = Report::affiliateTotal($date_from, $date_to, $offer_id, $country_id);
        $aff_sub_3 = Report::query()->distinct()->pluck('aff_sub_3')->toArray();
        $reports = [];
        foreach ($aff_sub_3 as $aff_sub_3) {
            $query = Report::where('aff_sub_3', $aff_sub_3)
                ->whereBetween('date', [$date_from, $date_to])
                ->where('user_id', Auth::id())
                ->when(!empty($offer_id), function ($query) use ($offer_id) {
                    return $query->whereIn('offer_id', $offer_id);
                });
            $click = $query->sum('click');
            $unique = $query->sum('unique');
            $conversion = $query->where('status', 'approved')->count();
            $revenue = $query->where('status', 'approved')->sum('revenue');
            $payout = $query->where('status', 'approved')->sum('payout');
            $profit = $query->where('status', 'approved')->sum('profit');
            $reports[$aff_sub_3] = compact('click', 'unique', 'conversion', 'revenue', 'payout', 'profit');
        }
        $countries = Country::all();
        $users = User::where('role', 'affiliate')->get();
        $advertisers = Advertiser::all();
        $offers = Offer::all();
        return view('affiliate.report.aff_sub_3', compact('countries', 'users', 'offers', 'advertisers', 'reports', 'total'));
    }

    public function click(Request $request)
    {
               // Default selected fields
               $selectedFields = [
                'offer' => $request->has('offer'),
                'country' => $request->has('country'),
                'device' => $request->has('device'),
                'os' => $request->has('os'),
                'browser' => $request->has('browser'),
                'click' => $request->has('click'),
                'unique' => $request->has('unique'),
                'payout' => $request->has('payout'),
                'epc' => $request->has('epc'),
                'conversions' => $request->has('conversions'),
            ];
    
            // Date range handling (same as before)
            if ($request->has('last_weak')) {
                $carbon_date_from = Carbon::now()->subWeek()->startOfWeek();
                $date_from = $carbon_date_from->toDateString();
                $carbon_date_to = Carbon::now()->subWeek()->endOfWeek();
                $date_to = $carbon_date_to->toDateString();
            } elseif ($request->has('this_weak')) {
                $carbon_date_from = Carbon::now()->startOfWeek();
                $date_from = $carbon_date_from->toDateString();
                $carbon_date_to = Carbon::now()->endOfWeek();
                $date_to = $carbon_date_to->toDateString();
            } elseif ($request->has('date_filter')) {
                $parts = explode(' - ', $request->date_filter);
                $date_from = $parts[0];
                $date_to = $parts[1];
            } else {
                $carbon_date_from = new Carbon('last Monday');
                $date_from = $carbon_date_from->toDateString();
                $carbon_date_to = new Carbon('this Sunday');
                $date_to = $carbon_date_to->toDateString();
            }
    
            // Collect filters from the request
            $offer_id = $request->get('offer_id', []);
            $country_id = $request->get('country_id', []);
            $advertiser_id = $request->get('advertiser_id', []);
    
            // Build the query based on selected fields
            $query = Report::with('offer', 'country')->whereBetween('date', [$date_from, $date_to])
                ->where('user_id', Auth::id())
                ->when(!empty($offer_id), function ($query) use ($offer_id) {
                    return $query->whereIn('offer_id', $offer_id);
                })
                ->when(!empty($country_id), function ($query) use ($country_id) {
                    return $query->whereIn('country_id', $country_id);
                })
                ->when(!empty($advertiser_id), function ($query) use ($advertiser_id) {
                    return $query->whereHas('advertiser', function ($query) use ($advertiser_id) {
                        return $query->whereIn('advertiser_id', $advertiser_id);
                    });
                });
    
            // dd(json_encode($query->get()));
    
            $reports = $query->groupBy('offer_id')
    ->selectRaw(
        'offer_id,' .
        'GROUP_CONCAT(DISTINCT countries.name) as country_name,' .
        'GROUP_CONCAT(DISTINCT os_name) as os_name,' .
        'GROUP_CONCAT(DISTINCT browser) as browser,' .
        'GROUP_CONCAT(DISTINCT user_agent) as user_agent,' .
        'GROUP_CONCAT(DISTINCT ip_address) as ip_address,' .
        'GROUP_CONCAT(DISTINCT aff_sub_1) as aff_sub_1,' .
        'GROUP_CONCAT(DISTINCT aff_sub_2) as aff_sub_2,' .
        'GROUP_CONCAT(DISTINCT aff_sub_3) as aff_sub_3,' .
        'GROUP_CONCAT(DISTINCT geo_city) as geo_city,' .
        'GROUP_CONCAT(DISTINCT geo_region) as geo_region,' .
        'SUM(click) as total_clicks,' .
        'SUM(`unique`) as total_uniques,' .
        'SUM(CASE WHEN status = "approved" THEN payout ELSE 0 END) as total_payouts,' .
        'COUNT(CASE WHEN status = "approved" THEN 1 END) as total_conversions,' .
        'AVG(CASE WHEN status = "approved" THEN payout / NULLIF(click, 0) ELSE NULL END) as epc'
    )
    ->leftJoin('countries', 'reports.country_id', '=', 'countries.id')
    ->get();

        
       // dd($reports->toArray());
            // Preparing total and selected fields for display
            $countries = Country::all();
            $users = User::where('role', 'affiliate')->get();
            $advertisers = Advertiser::all();
            $offers = Offer::all();
            $total_clicks = $reports->sum('total_clicks'); // Replace with actual calculation logic
            $total_uniques = $reports->sum('total_uniques');
            $total_conversions = $reports->sum('total_conversions');
            $total_payouts = $reports->sum('total_payouts');
            $total_epc = $reports->sum('epc');
    
            // Prepare totals array for returning to the view
            $total = [
                'clicks' => $total_clicks,
                'uniques' => $total_uniques,
                'conversion' => $total_conversions,
                'payouts' => $total_payouts,
                'epc' => $total_epc,
            ];
            
            // Return to view with the selected fields
            return view('affiliate.report.click', compact('countries', 'users', 'offers', 'advertisers', 'reports', 'selectedFields', 'total'));
       
    }

    public function conversion(Request $request)
    {
                      // Default selected fields
                      $selectedFields = [
                        'offer' => $request->has('offer'),
                        'country' => $request->has('country'),
                        'device' => $request->has('device'),
                        'os' => $request->has('os'),
                        'browser' => $request->has('browser'),
                        'click' => $request->has('click'),
                        'unique' => $request->has('unique'),
                        'payout' => $request->has('payout'),
                        'epc' => $request->has('epc'),
                        'conversions' => $request->has('conversions'),
                    ];
            
                    // Date range handling (same as before)
                    if ($request->has('last_weak')) {
                        $carbon_date_from = Carbon::now()->subWeek()->startOfWeek();
                        $date_from = $carbon_date_from->toDateString();
                        $carbon_date_to = Carbon::now()->subWeek()->endOfWeek();
                        $date_to = $carbon_date_to->toDateString();
                    } elseif ($request->has('this_weak')) {
                        $carbon_date_from = Carbon::now()->startOfWeek();
                        $date_from = $carbon_date_from->toDateString();
                        $carbon_date_to = Carbon::now()->endOfWeek();
                        $date_to = $carbon_date_to->toDateString();
                    } elseif ($request->has('date_filter')) {
                        $parts = explode(' - ', $request->date_filter);
                        $date_from = $parts[0];
                        $date_to = $parts[1];
                    } else {
                        $carbon_date_from = new Carbon('last Monday');
                        $date_from = $carbon_date_from->toDateString();
                        $carbon_date_to = new Carbon('this Sunday');
                        $date_to = $carbon_date_to->toDateString();
                    }
            
                    // Collect filters from the request
                    $offer_id = $request->get('offer_id', []);
                    $country_id = $request->get('country_id', []);
                    $advertiser_id = $request->get('advertiser_id', []);
            
                    // Build the query based on selected fields
                    $query = Report::with('offer', 'country')->whereBetween('date', [$date_from, $date_to])
                        ->where('user_id', Auth::id())
                        ->when(!empty($offer_id), function ($query) use ($offer_id) {
                            return $query->whereIn('offer_id', $offer_id);
                        })
                        ->when(!empty($country_id), function ($query) use ($country_id) {
                            return $query->whereIn('country_id', $country_id);
                        })
                        ->when(!empty($advertiser_id), function ($query) use ($advertiser_id) {
                            return $query->whereHas('advertiser', function ($query) use ($advertiser_id) {
                                return $query->whereIn('advertiser_id', $advertiser_id);
                            });
                        });
            
                    // dd(json_encode($query->get()));
            
                    $reports = $query->groupBy('offer_id')
                    ->selectRaw(
                        'offer_id,' .
                        'GROUP_CONCAT(DISTINCT countries.name) as country_name,' .  // Group concatenation for country_name
                        'GROUP_CONCAT(DISTINCT os_name) as os_name,' .              // Group concatenation for os_name
                        'GROUP_CONCAT(DISTINCT browser) as browser,' .              // Group concatenation for browser
                        'GROUP_CONCAT(DISTINCT user_agent) as user_agent,' .        // Group concatenation for user_agent
                        'GROUP_CONCAT(DISTINCT ip_address) as ip_address,' .        // Group concatenation for ip_address
                        'GROUP_CONCAT(DISTINCT aff_sub_1) as aff_sub_1,' .          // Group concatenation for aff_sub_1
                        'GROUP_CONCAT(DISTINCT aff_sub_2) as aff_sub_2,' .          // Group concatenation for aff_sub_2
                        'GROUP_CONCAT(DISTINCT aff_sub_3) as aff_sub_3,' .           // Group concatenation for aff_sub_3
                        'GROUP_CONCAT(DISTINCT geo_city) as geo_city,' .           // Group concatenation for aff_sub_3
                        'GROUP_CONCAT(DISTINCT geo_region) as geo_region,' .           // Group concatenation for aff_sub_3
                        'SUM(click) as total_clicks,' .
        'SUM(`unique`) as total_uniques,' .
        'SUM(CASE WHEN status = "approved" THEN payout ELSE 0 END) as total_payouts,' .
        'COUNT(CASE WHEN status = "approved" THEN 1 END) as total_conversions,' .
        'AVG(CASE WHEN status = "approved" THEN payout / NULLIF(click, 0) ELSE NULL END) as epc'
                    )
                    ->leftJoin('countries', 'reports.country_id', '=', 'countries.id')
                    ->get();
                
                
                    // Preparing total and selected fields for display
                    $countries = Country::all();
                    $users = User::where('role', 'affiliate')->get();
                    $advertisers = Advertiser::all();
                    $offers = Offer::all();
                    $total_clicks = $reports->sum('total_clicks'); // Replace with actual calculation logic
                    $total_uniques = $reports->sum('total_uniques');
                    $total_conversions = $reports->sum('total_conversions');
                    $total_payouts = $reports->sum('total_payouts');
                    $total_epc = $reports->sum('epc');
            
                    // Prepare totals array for returning to the view
                    $total = [
                        'clicks' => $total_clicks,
                        'uniques' => $total_uniques,
                        'conversion' => $total_conversions,
                        'payouts' => $total_payouts,
                        'epc' => $total_epc,
                    ];
            
                    // Return to view with the selected fields
                    return view('affiliate.report.conversion', compact('countries', 'users', 'offers', 'advertisers', 'reports', 'selectedFields', 'total'));
    }

}
