<?php

namespace App\Http\Controllers;

use App\Chat;
use App\Message;
use App\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Pusher\Pusher;

class ChatController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display the chat interface.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $chats = $user->chats()->with(['participants', 'messages' => function($query) {
            $query->latest()->limit(1);
        }])->get();

        // Get users for starting new chats (only managers or users under current manager)
        $users = $this->getAvailableUsers($user);

        // If manager_id is provided, automatically start/select chat with that manager
        $selectedChatId = null;
        if ($request->has('manager_id')) {
            $managerId = $request->get('manager_id');
            if ($this->canChatWith($user, User::find($managerId))) {
                $chat = $this->getOrCreateChatWithUser($managerId);
                $selectedChatId = $chat->id;
            }
        }

        // Determine layout based on user role
        $layout = 'layouts.backend.affiliate_sash'; // default
        if ($user->role === 'admin') {
            $layout = 'layouts.backend.admin_sash';
        } elseif ($user->role === 'manager') {
            $layout = 'layouts.backend.manager_sash';
        }

        return view('chat.index', compact('chats', 'users', 'selectedChatId', 'layout'));
    }

    /**
     * Get chat data for dashboard card.
     */
    public function getChatData()
    {
        $user = Auth::user();
        $chats = $user->chats()->with(['participants', 'messages' => function($query) {
            $query->latest()->limit(1);
        }])->get();

        // Get users for starting new chats (only managers or users under current manager)
        $users = $this->getAvailableUsers($user);

        return response()->json([
            'success' => true,
            'chats' => $chats,
            'users' => $users
        ]);
    }

    /**
     * Get available users for chat based on manager relationships.
     */
    private function getAvailableUsers($user)
    {
        $availableUsers = collect();

        if ($user->role === 'manager') {
            // Manager can chat with their users
            $availableUsers = $user->users()->get();
        } elseif ($user->role === 'admin') {
            // Admin can chat with managers
            $availableUsers = User::where('role', 'manager')->get();
        } else {
            // Regular user can chat with their manager
            if ($user->manager_id) {
                $availableUsers = User::where('id', $user->manager_id)->get();
            }
        }

        return $availableUsers;
    }

    /**
     * Get or create a private chat between two users.
     */
    public function getOrCreateChat(Request $request)
    {
        $currentUserId = Auth::id();
        $otherUserId = $request->input('user_id');

        if ($currentUserId === $otherUserId) {
            return response()->json(['error' => 'Cannot chat with yourself'], 400);
        }

        // Validate that the user can chat with the selected user
        $currentUser = Auth::user();
        $otherUser = User::find($otherUserId);
        
        if (!$this->canChatWith($currentUser, $otherUser)) {
            return response()->json(['error' => 'You can only chat with your manager or assigned users'], 403);
        }

        // Check if chat already exists between these two users
        $existingChat = Chat::whereHas('participants', function($query) use ($currentUserId) {
            $query->where('user_id', $currentUserId);
        })->whereHas('participants', function($query) use ($otherUserId) {
            $query->where('user_id', $otherUserId);
        })->where('type', 'private')->first();

        if ($existingChat) {
            return response()->json([
                'success' => true,
                'chat' => $existingChat->load(['participants', 'messages.user'])
            ]);
        }

        // Create new chat
        DB::beginTransaction();
        try {
            $chat = Chat::create([
                'type' => 'private'
            ]);

            // Add participants
            $chat->participants()->attach([$currentUserId, $otherUserId]);

            DB::commit();

            return response()->json([
                'success' => true,
                'chat' => $chat->load(['participants', 'messages.user'])
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(['error' => 'Failed to create chat'], 500);
        }
    }

    /**
     * Check if current user can chat with the other user.
     */
    private function canChatWith($currentUser, $otherUser)
    {
        if (!$otherUser) {
            return false;
        }

        if ($currentUser->role === 'manager') {
            // Manager can chat with their users
            return $otherUser->manager_id === $currentUser->id;
        } elseif ($currentUser->role === 'admin') {
            // Admin can chat with managers
            return $otherUser->role === 'manager';
        } else {
            // Regular user can chat with their manager
            return $otherUser->id === $currentUser->manager_id;
        }
    }

    /**
     * Send a message.
     */
    public function sendMessage(Request $request)
    {
        \Log::info('Send message request:', $request->all());
        
        $request->validate([
            'chat_id' => 'required|exists:chats,id',
            'message' => 'required|string|max:1000',
            'type' => 'sometimes|string|in:text,image,file'
        ]);

        $user = Auth::user();
        $chat = Chat::findOrFail($request->chat_id);

        // Check if user is participant of this chat
        if (!$chat->hasParticipant($user->id)) {
            \Log::error('User not participant of chat', ['user_id' => $user->id, 'chat_id' => $chat->id]);
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $message = Message::create([
            'chat_id' => $chat->id,
            'user_id' => $user->id,
            'message' => $request->message,
            'type' => $request->input('type', 'text'),
            'metadata' => $request->input('metadata')
        ]);

        $message->load('user');

        // Broadcast message to Pusher
        try {
            $this->broadcastMessage($message, $chat);
        } catch (\Exception $e) {
            \Log::error('Broadcast failed but message saved: ' . $e->getMessage());
        }

        \Log::info('Message sent successfully', ['message_id' => $message->id]);
        
        return response()->json([
            'success' => true,
            'message' => $message
        ]);
    }

    /**
     * Get messages for a chat.
     */
    public function getMessages(Request $request, $chatId)
    {
        $user = Auth::user();
        $chat = Chat::findOrFail($chatId);

        // Check if user is participant of this chat
        if (!$chat->hasParticipant($user->id)) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $messages = $chat->messages()
            ->with('user')
            ->orderBy('created_at', 'asc')
            ->get();

        // Mark messages as read
        $chat->participants()->updateExistingPivot($user->id, [
            'last_read_at' => now()
        ]);

        return response()->json([
            'success' => true,
            'messages' => $messages
        ]);
    }

    /**
     * Get chat details.
     */
    public function getChat($chatId)
    {
        $user = Auth::user();
        $chat = Chat::findOrFail($chatId);

        // Check if user is participant of this chat
        if (!$chat->hasParticipant($user->id)) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $chat->load(['participants', 'messages.user']);

        return response()->json([
            'success' => true,
            'chat' => $chat
        ]);
    }

    /**
     * Mark messages as read.
     */
    public function markAsRead(Request $request, $chatId)
    {
        $user = Auth::user();
        $chat = Chat::findOrFail($chatId);

        // Check if user is participant of this chat
        if (!$chat->hasParticipant($user->id)) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $chat->participants()->updateExistingPivot($user->id, [
            'last_read_at' => now()
        ]);

        return response()->json(['success' => true]);
    }

    /**
     * Broadcast message using Pusher.
     */
    private function broadcastMessage(Message $message, Chat $chat)
    {
        try {
            $pusher = new Pusher(
                config('broadcasting.connections.pusher.key'),
                config('broadcasting.connections.pusher.secret'),
                config('broadcasting.connections.pusher.app_id'),
                config('broadcasting.connections.pusher.options')
            );

            $pusher->trigger(
                'private-chat.' . $chat->id,
                'message.sent',
                [
                    'message' => $message,
                    'chat_id' => $chat->id
                ]
            );
        } catch (\Exception $e) {
            \Log::error('Pusher broadcast failed: ' . $e->getMessage());
        }
    }

    /**
     * Get Pusher authentication for private channels.
     */
    public function pusherAuth(Request $request)
    {
        $user = Auth::user();
        $socketId = $request->input('socket_id');
        $channel = $request->input('channel_name');

        // Validate channel name format
        if (!preg_match('/^private-chat\.\d+$/', $channel)) {
            return response()->json(['error' => 'Invalid channel'], 400);
        }

        // Extract chat ID from channel name
        $chatId = (int) substr($channel, strrpos($channel, '.') + 1);
        $chat = Chat::find($chatId);

        if (!$chat || !$chat->hasParticipant($user->id)) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        try {
            $pusher = new Pusher(
                config('broadcasting.connections.pusher.key'),
                config('broadcasting.connections.pusher.secret'),
                config('broadcasting.connections.pusher.app_id'),
                config('broadcasting.connections.pusher.options')
            );

            $auth = $pusher->authorizeChannel($channel, $socketId);

            return response($auth);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Authentication failed'], 500);
        }
    }

    /**
     * Get or create chat with a specific user.
     */
    private function getOrCreateChatWithUser($userId)
    {
        $currentUser = Auth::user();
        $otherUser = User::findOrFail($userId);

        // Check if chat already exists
        $existingChat = Chat::whereHas('participants', function($query) use ($currentUser) {
            $query->where('user_id', $currentUser->id);
        })->whereHas('participants', function($query) use ($otherUser) {
            $query->where('user_id', $otherUser->id);
        })->first();

        if ($existingChat) {
            return $existingChat;
        }

        // Create new chat
        $chat = Chat::create([
            'name' => null,
            'type' => 'private'
        ]);

        // Add participants
        $chat->participants()->attach($currentUser->id);
        $chat->participants()->attach($otherUser->id);

        return $chat;
    }
}
