<?php

namespace App\Http\Controllers\Manager;

use App\User;
use App\Offer;
use App\Report;
use Carbon\Carbon;
use App\OfferApplication;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        $user_id = User::where('manager_id', Auth::user()->id)->pluck('id')->toArray();
        
        // Handle date range for main chart - default to last 7 days
        if (isset($request->date_filter)) {
            $parts = explode(' - ', $request->date_filter);
            $date_from = $parts[0];
            $date_to = $parts[1];
        } else {
            // Default to last 7 days instead of weekly range
            $carbon_date_from = Carbon::now()->subDays(6)->startOfDay();
            $date_from = $carbon_date_from->toDateString();
            $carbon_date_to = Carbon::now()->endOfDay();
            $date_to = $carbon_date_to->toDateString();
        }
        $period = \Carbon\CarbonPeriod::create($date_from, $date_to);

        $reports = [];

        foreach ($period as $date) {

            $click = Report::where('date', $date->toDateString())
                ->whereIn('user_id', $user_id)
                ->get()
                ->sum('click');

            $unique = Report::where('date', $date->toDateString())
                ->whereIn('user_id', $user_id)
                ->get()
                ->sum('unique');

            $conversion = Report::where('date', $date->toDateString())
                ->where('status', 'approved')
                ->whereIn('user_id', $user_id)
                ->get()
                ->count();

            $payout = Report::where('date', $date->toDateString())
                ->where('status', 'approved')
                ->whereIn('user_id', $user_id)
                ->whereNotNull('payout')
                ->where('payout', '>', 0)
                ->get()
                ->sum('payout');


            $reports[$date->toDateString()] = compact('click', 'unique', 'conversion', 'payout');
        }

        // Get hourly stats for stats cards (today's hourly data)
        $hourlyStats = [];
        $today = Carbon::now()->startOfDay();
        
        // Get today's hourly data using created_at
        for ($i = 0; $i < 24; $i++) {
            $hour = sprintf('%02d:00', $i);
            $hourStart = $today->copy()->addHours($i);
            $hourEnd = $hourStart->copy()->addHour();
            
            $hourlyClick = Report::whereBetween('created_at', [$hourStart, $hourEnd])
                ->whereIn('user_id', $user_id)
                ->sum('click');
            
            $hourlyUnique = Report::whereBetween('created_at', [$hourStart, $hourEnd])
                ->whereIn('user_id', $user_id)
                ->sum('unique');
            
            $hourlyConversion = Report::whereBetween('updated_at', [$hourStart, $hourEnd])
                ->where('status','approved')
                ->whereIn('user_id', $user_id)
                ->count();
            
            $hourlyPayout = Report::whereBetween('updated_at', [$hourStart, $hourEnd])
                ->where('status','approved')
                ->whereIn('user_id', $user_id)
                ->whereNotNull('payout')
                ->where('payout', '>', 0)
                ->sum('payout');
            
            $hourlyStats[$hour] = compact('hourlyClick','hourlyUnique','hourlyConversion','hourlyPayout');
        }
        $totalClick = Report::where('date', '>=', $date_from)
            ->whereIn('user_id', $user_id)
            ->where('date', '<=', $date_to)
            ->sum('click');
        $totalUnique = Report::where('date', '>=', $date_from)
            ->whereIn('user_id', $user_id)
            ->sum('unique');
        $totalConversion = Report::where('date', '>=', $date_from)
            ->whereIn('user_id', $user_id)
            ->where('date', '<=', $date_to)
            ->where('status', 'approved')
            ->count();
        $totalPayout = Report::where('date', '>=', $date_from)
            ->whereIn('user_id', $user_id)
            ->where('date', '<=', $date_to)
            ->where('status', 'approved')
            ->whereNotNull('payout')
            ->where('payout', '>', 0)
            ->sum('payout');
        // Top Tasks
        $top_offers =  Offer::withCount([
            'reports' => function ($query) use ($date_from, $date_to, $user_id) {
                $query->where('date', '>=', $date_from)->where('date', '<=', $date_to)->whereIn('user_id', $user_id)->where('status', 'approved');
            }
        ])
            ->orderByDesc("reports_count")
            ->take(5)
            ->get();
        $top_countries = Report::with('country')
            ->where('status', 'approved')
            ->where('date', '>=', $date_from)
            ->whereIn('user_id', $user_id)
            ->where('date', '<=', $date_to)
            ->select('country_id', DB::raw('count(*) as total'))
            ->groupBy('country_id')
            ->whereIn('user_id', $user_id)
            ->get();
        $top_users =  User::whereIn('id',$user_id)->where('status', 'active')->where('role', 'affiliate')->withCount([
            'reports' => function ($query) use ($date_from, $date_to, $user_id) {
                $query->where('date', '>=', $date_from)->where('date', '<=', $date_to)->whereIn('user_id', $user_id)->where('status', 'approved');
            }
        ])
            ->orderByDesc("reports_count")
            ->take(5)
            ->get();
        // Last Month Data
        $last_month = [];
        $startMonth = Carbon::now()->subMonth()->startOfMonth()->toDateString();
        $endMonth = Carbon::now()->subMonth()->endOfMonth()->toDateString();
        $last_month_click = Report::whereBetween('date', [$startMonth, $endMonth])->whereIn('user_id',$user_id)->where('status', 'click')->count();
        $last_month_conversion = Report::whereBetween('date', [$startMonth, $endMonth])->whereIn('user_id',$user_id)->where('status', 'approved')->count();
        $last_month_payout = Report::whereBetween('date', [$startMonth, $endMonth])->whereIn('user_id',$user_id)->where('status', 'approved')->whereNotNull('payout')->where('payout', '>', 0)->sum('payout');
        $last_month_profit = Report::whereBetween('date', [$startMonth, $endMonth])->whereIn('user_id',$user_id)->where('status', 'approved')->sum('profit');
        $last_month = compact('last_month_click', 'last_month_conversion', 'last_month_payout', 'last_month_profit');

        $pending_affiliates = User::where('status', 'pending')->whereIn('id',$user_id)->where('role', 'affiliate')->take(10)
            ->get();
        $offer_applications = OfferApplication::where('status', 'false')->take(10)->whereIn('user_id', $user_id)
            ->get();
            
        // Prepare chart data for Analytics Overview (same format as admin)
        $chartData = [];
        foreach ($period as $date) {
            $dateStr = $date->toDateString();
            $dayData = $reports[$dateStr] ?? [];
            $chartData[] = [
                'date' => $dateStr,
                'clicks' => $dayData['click'] ?? 0,
                'conversions' => $dayData['conversion'] ?? 0,
                'payout' => $dayData['payout'] ?? 0,
            ];
        }
        
        // Convert hourlyStats to timeBasedStats format (same as admin)
        $timeBasedStats = [];
        foreach ($hourlyStats as $hour => $data) {
            $timeBasedStats[$hour] = [
                'timeClick' => $data['hourlyClick'] ?? 0,
                'timeUnique' => $data['hourlyUnique'] ?? 0,
                'timeConversion' => $data['hourlyConversion'] ?? 0,
                'timePayout' => $data['hourlyPayout'] ?? 0,
            ];
        }
        
        return view('manager.dashboard', compact('reports', 'hourlyStats', 'chartData', 'timeBasedStats', 'totalClick', 'totalUnique', 'totalConversion','totalPayout', 'top_offers', 'top_countries', 'top_users', 'pending_affiliates', 'offer_applications', 'last_month'));
    }
}
