@extends('layouts.backend.manager_sash')

@section('title', 'Chat')

@section('content')
<style>
/* Modern Chat Interface Styles */
.chat-container {
    height: calc(100vh - 200px);
    min-height: 600px;
    background: #f8fafc;
    border-radius: 16px;
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    overflow: hidden;
}

.chat-sidebar {
    width: 320px;
    background: white;
    border-right: 1px solid #e2e8f0;
    display: flex;
    flex-direction: column;
    transition: all 0.3s ease;
}

.chat-sidebar-header {
    padding: 20px;
    border-bottom: 1px solid #e2e8f0;
    background: white;
}

.chat-search-container {
    position: relative;
    margin-bottom: 16px;
}

.chat-search-input {
    width: 100%;
    padding: 12px 16px 12px 40px;
    border: 1px solid #e2e8f0;
    border-radius: 12px;
    background: #f8fafc;
    font-size: 14px;
    transition: all 0.2s ease;
}

.chat-search-input:focus {
    outline: none;
    border-color: #3b82f6;
    background: white;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.chat-search-icon {
    position: absolute;
    left: 12px;
    top: 50%;
    transform: translateY(-50%);
    color: #64748b;
    font-size: 16px;
}

.sidebar-title {
    font-size: 18px;
    font-weight: 600;
    color: #1e293b;
    margin: 0;
}

.new-chat-btn {
    background: #3b82f6;
    color: white;
    border: none;
    padding: 8px 12px;
    border-radius: 8px;
    font-size: 14px;
    font-weight: 500;
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    gap: 6px;
}

.new-chat-btn:hover {
    background: #2563eb;
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(59, 130, 246, 0.3);
}

.chat-list-container {
    flex: 1;
    overflow-y: auto;
    padding: 8px;
}

.chat-item {
    padding: 16px;
    border-radius: 12px;
    margin-bottom: 4px;
    cursor: pointer;
    transition: all 0.2s ease;
    border: 1px solid transparent;
}

.chat-item:hover {
    background: #f1f5f9;
    transform: translateX(2px);
}

.chat-item.active {
    background: #eff6ff;
    border-color: #3b82f6;
}

.chat-item-avatar {
    width: 48px;
    height: 48px;
    border-radius: 50%;
    object-fit: cover;
}

.chat-item-content {
    flex: 1;
    min-width: 0;
}

.chat-item-name {
    font-size: 15px;
    font-weight: 600;
    color: #1e293b;
    margin: 0 0 4px 0;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.chat-item-preview {
    font-size: 13px;
    color: #64748b;
    margin: 0;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.chat-item-time {
    font-size: 12px;
    color: #94a3b8;
    white-space: nowrap;
}

.chat-main-area {
    flex: 1;
    display: flex;
    flex-direction: column;
    background: #f8fafc;
    position: relative;
    height: 100%;
    overflow: hidden;
}

.no-chat-selected {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    height: 100%;
    text-align: center;
    padding: 40px;
}

/* Chat Messages Area */
#chatMessages {
    display: flex;
    flex-direction: column;
    height: 100%;
    overflow: hidden;
}

.no-chat-icon {
    font-size: 64px;
    color: #cbd5e1;
    margin-bottom: 24px;
}

.no-chat-title {
    font-size: 24px;
    font-weight: 600;
    color: #1e293b;
    margin-bottom: 12px;
}

.no-chat-description {
    font-size: 16px;
    color: #64748b;
    margin-bottom: 32px;
    max-width: 400px;
}

.start-chat-btn {
    background: #3b82f6;
    color: white;
    border: none;
    padding: 12px 24px;
    border-radius: 12px;
    font-size: 16px;
    font-weight: 500;
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    gap: 8px;
}

.start-chat-btn:hover {
    background: #2563eb;
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(59, 130, 246, 0.3);
}

.chat-header {
    background: white;
    padding: 20px 24px;
    border-bottom: 1px solid #e2e8f0;
    display: flex;
    align-items: center;
    justify-content: space-between;
    flex-shrink: 0; /* Prevent shrinking */
    height: 80px; /* Fixed height */
    min-height: 80px;
    max-height: 80px;
}

.chat-header-user {
    display: flex;
    align-items: center;
    gap: 12px;
}

.chat-header-avatar {
    width: 48px;
    height: 48px;
    border-radius: 50%;
    object-fit: cover;
}

.chat-header-info h6 {
    font-size: 16px;
    font-weight: 600;
    color: #1e293b;
    margin: 0;
}

.chat-header-info small {
    font-size: 13px;
    color: #64748b;
}

.chat-info-btn {
    background: #f1f5f9;
    border: 1px solid #e2e8f0;
    color: #64748b;
    padding: 8px;
    border-radius: 8px;
    transition: all 0.2s ease;
}

.chat-info-btn:hover {
    background: #e2e8f0;
    color: #475569;
}

.messages-container {
    flex: 1;
    overflow-y: auto;
    padding: 24px;
    background: #f8fafc;
    min-height: 0; /* Allow flex item to shrink */
    max-height: calc(100% - 160px); /* Reserve space for header (80px) and input (80px) */
}

.message-bubble {
    max-width: 70%;
    padding: 12px 16px;
    border-radius: 18px;
    margin-bottom: 16px;
    position: relative;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
}

.message-bubble.own {
    background: #3b82f6;
    color: white;
    margin-left: auto;
}

.message-bubble.other {
    background: white;
    color: #1e293b;
    border: 1px solid #e2e8f0;
}

.message-text {
    margin: 0 0 4px 0;
    font-size: 14px;
    line-height: 1.4;
}

.message-time {
    font-size: 11px;
    opacity: 0.7;
}

.message-input-container {
    background: white;
    padding: 20px 24px;
    border-top: 1px solid #e2e8f0;
    display: flex;
    align-items: center;
    gap: 12px;
    flex-shrink: 0; /* Prevent shrinking */
    height: 80px; /* Fixed height */
    min-height: 80px; /* Ensure minimum height */
    max-height: 80px; /* Prevent growing */
}

.message-input {
    flex: 1;
    padding: 12px 16px;
    border: 1px solid #e2e8f0;
    border-radius: 24px;
    font-size: 14px;
    background: #f8fafc;
    transition: all 0.2s ease;
}

.message-input:focus {
    outline: none;
    border-color: #3b82f6;
    background: white;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.send-btn {
    background: #3b82f6;
    color: white;
    border: none;
    padding: 12px;
    border-radius: 50%;
    width: 48px;
    height: 48px;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.2s ease;
}

.send-btn:hover {
    background: #2563eb;
    transform: scale(1.05);
}

.send-btn:disabled {
    background: #cbd5e1;
    cursor: not-allowed;
    transform: none;
}

/* Mobile Responsive */
@media (max-width: 768px) {
    .chat-container {
        height: calc(100vh - 120px);
        border-radius: 0;
    }
    
    .chat-sidebar {
        position: absolute;
        top: 0;
        left: 0;
        height: 100%;
        z-index: 1000;
        transform: translateX(-100%);
        box-shadow: 4px 0 6px -1px rgba(0, 0, 0, 0.1);
    }
    
    .chat-sidebar.open {
        transform: translateX(0);
    }
    
    .chat-main-area {
        width: 100%;
    }
    
    .mobile-sidebar-toggle {
        display: block;
        background: #3b82f6;
        color: white;
        border: none;
        padding: 8px 12px;
        border-radius: 8px;
        margin-right: 12px;
    }
    
    .message-bubble {
        max-width: 85%;
    }
    
    .messages-container {
        padding: 16px;
    }
    
    .message-input-container {
        padding: 16px;
    }
}

@media (min-width: 769px) {
    .mobile-sidebar-toggle {
        display: none;
    }
}

/* Scrollbar Styling */
.chat-list-container::-webkit-scrollbar,
.messages-container::-webkit-scrollbar {
    width: 6px;
}

.chat-list-container::-webkit-scrollbar-track,
.messages-container::-webkit-scrollbar-track {
    background: #f1f5f9;
}

.chat-list-container::-webkit-scrollbar-thumb,
.messages-container::-webkit-scrollbar-thumb {
    background: #cbd5e1;
    border-radius: 3px;
}

.chat-list-container::-webkit-scrollbar-thumb:hover,
.messages-container::-webkit-scrollbar-thumb:hover {
    background: #94a3b8;
}
</style>

<div class="row">
    <div class="col-12">
        <div class="card border-0 shadow-sm">
            <div class="card-header bg-white border-0 py-3">
                <div class="d-flex justify-content-between align-items-center">
                    <div class="d-flex align-items-center">
                        <button class="mobile-sidebar-toggle" id="mobileSidebarToggle">
                            <i class="fe fe-menu"></i>
                        </button>
                        <h3 class="card-title mb-0">
                            <i class="fe fe-message-circle me-2 text-primary"></i>Live Chat
                        </h3>
                    </div>
                    <div class="d-flex align-items-center">
                        <span class="badge bg-success-transparent me-2" id="chatStatus">
                            <i class="fe fe-circle me-1" style="font-size: 0.5rem;"></i>Online
                        </span>
                    </div>
                </div>
            </div>
            <div class="card-body p-0">
                <div class="d-flex chat-container" id="chatContainer">
                    
                    <!-- Chat Sidebar -->
                    <div class="chat-sidebar" id="chatSidebar">
                        <div class="chat-sidebar-header">
                            <div class="chat-search-container">
                                <i class="fe fe-search chat-search-icon"></i>
                                <input type="text" class="chat-search-input" placeholder="Search conversations..." id="chatSearchInput">
                            </div>
                            <div class="d-flex justify-content-between align-items-center">
                                <h6 class="sidebar-title">Conversations</h6>
                                <button class="new-chat-btn" id="chatWithUserBtn" title="Start new chat">
                                    <i class="fe fe-plus"></i>
                                    <span class="d-none d-md-inline">New Chat</span>
                                </button>
                            </div>
                        </div>
                        
                        <div class="chat-list-container" id="chatList">
                            <!-- Chat list will be populated here -->
                        </div>
                    </div>
                    
                    <!-- Chat Main Area -->
                    <div class="chat-main-area">
                        <!-- No Chat Selected State -->
                        <div id="noChatSelected" class="no-chat-selected">
                            <div class="no-chat-icon">
                                <i class="fe fe-message-circle"></i>
                            </div>
                            <h5 class="no-chat-title">Welcome to Live Chat</h5>
                            <p class="no-chat-description">Select a conversation from the sidebar or start a new chat with your affiliates.</p>
                            <button class="start-chat-btn" id="startNewChatBtn">
                                <i class="fe fe-plus"></i>Start New Chat
                            </button>
                        </div>
                        
                        <!-- Chat Messages Area -->
                        <div id="chatMessages" class="d-none flex-grow-1 d-flex flex-column">
                            <!-- Chat Header -->
                            <div class="chat-header">
                                <div class="chat-header-user">
                                    <img src="" alt="" class="chat-header-avatar" id="chatUserAvatar">
                                    <div class="chat-header-info">
                                        <h6 id="chatUserName"></h6>
                                        <small id="chatUserStatus">Online</small>
                                    </div>
                                </div>
                                <div class="d-flex align-items-center">
                                    <button class="chat-info-btn" id="chatInfoBtn" title="Chat Info">
                                        <i class="fe fe-info"></i>
                                    </button>
                                </div>
                            </div>
                            
                            <!-- Messages Container -->
                            <div class="messages-container" id="messagesContainer">
                                <div id="messagesList">
                                    <!-- Messages will be populated here -->
                                </div>
                            </div>
                            
                            <!-- Message Input -->
                            <div class="message-input-container">
                                <input type="text" class="message-input" id="messageInput" placeholder="Type your message...">
                                <button class="send-btn" id="sendButton">
                                    <i class="fe fe-send"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Start New Chat Modal -->
<div class="modal fade" id="startChatModal" tabindex="-1" aria-labelledby="startChatModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg modal-dialog-centered">
        <div class="modal-content border-0 shadow-lg" style="border-radius: 16px;">
            <div class="modal-header border-0 pb-0" style="padding: 24px 24px 0 24px;">
                <h5 class="modal-title fw-bold" style="color: #1e293b; font-size: 20px;">Start a New Chat</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close" style="background: #f1f5f9; border-radius: 8px; padding: 8px;"></button>
            </div>
            <div class="modal-body" style="padding: 24px;">
                <div class="mb-4">
                    <label class="form-label fw-semibold mb-3" style="color: #1e293b; font-size: 14px;">Select Affiliate to Chat With:</label>
                    <select class="form-select" id="userSelect" style="border: 1px solid #e2e8f0; border-radius: 12px; padding: 12px 16px; background: #f8fafc;">
                        <option value="">Choose an affiliate...</option>
                    </select>
                </div>
                <div id="selectedUserInfo" class="p-4 border rounded-3" style="background: #f8fafc; border-color: #e2e8f0; display: none;">
                    <div class="d-flex align-items-center">
                        <img src="" alt="" class="rounded-circle me-3" id="selectedUserAvatar" style="width: 48px; height: 48px; object-fit: cover;">
                        <div>
                            <h6 class="mb-1 fw-bold" id="selectedUserName" style="color: #1e293b; font-size: 16px;"></h6>
                            <small class="text-muted" id="selectedUserEmail" style="font-size: 13px;"></small>
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-footer border-0 pt-0" style="padding: 0 24px 24px 24px;">
                <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal" style="border: 1px solid #e2e8f0; color: #64748b; padding: 10px 20px; border-radius: 10px; font-weight: 500;">Cancel</button>
                <button type="button" class="btn btn-primary" id="startChatBtn" disabled style="background: #3b82f6; border: none; padding: 10px 20px; border-radius: 10px; font-weight: 500;">Start Chat</button>
            </div>
        </div>
    </div>
</div>

@endsection

@push('js')
<script>
let currentChatId = null;
let pusher = null;
let currentChannel = null;

// Initialize Pusher
function initializePusher() {
    try {
        pusher = new Pusher('{{ config('broadcasting.connections.pusher.key') }}', {
            cluster: '{{ config('broadcasting.connections.pusher.options.cluster') }}',
            encrypted: true,
            authEndpoint: '{{ route("manager.chat.pusher-auth") }}',
            auth: {
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            }
        });
        
        console.log('Pusher initialized successfully');
    } catch (error) {
        console.error('Failed to initialize Pusher:', error);
    }
}

// Load chat data
async function loadChatData() {
    try {
        const response = await fetch('{{ route("manager.chat.data") }}');
        const data = await response.json();
        
        if (data.success) {
            renderChatList(data.chats);
            populateUserSelect(data.users);
        }
    } catch (error) {
        console.error('Error loading chat data:', error);
    }
}

// Render chat list
function renderChatList(chats) {
    const chatList = document.getElementById('chatList');
    
    if (chats.length === 0) {
        chatList.innerHTML = `
            <div class="text-center p-4">
                <i class="fe fe-message-circle fs-2 text-muted mb-3"></i>
                <p class="text-muted mb-0">No conversations yet</p>
                <small class="text-muted">Start chatting with your affiliates</small>
            </div>
        `;
        return;
    }
    
    chatList.innerHTML = chats.map(chat => {
        const otherParticipant = chat.participants.find(p => p.id !== {{ auth()->id() }});
        const lastMessage = chat.messages[0];
        
        return `
            <div class="chat-item" data-chat-id="${chat.id}" onclick="selectChat(${chat.id})">
                <div class="d-flex align-items-center">
                    <img src="${otherParticipant.avatar_url || 'https://ui-avatars.com/api/?name=' + encodeURIComponent(otherParticipant.first_name + ' ' + otherParticipant.last_name) + '&background=3b82f6&color=fff'}" alt="" class="chat-item-avatar me-3">
                    <div class="chat-item-content">
                        <h6 class="chat-item-name">${otherParticipant.first_name} ${otherParticipant.last_name}</h6>
                        <p class="chat-item-preview">${lastMessage ? lastMessage.message.substring(0, 50) + (lastMessage.message.length > 50 ? '...' : '') : 'No messages yet'}</p>
                    </div>
                    <div class="chat-item-time">
                        ${lastMessage ? new Date(lastMessage.created_at).toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'}) : ''}
                    </div>
                </div>
            </div>
        `;
    }).join('');
}

// Populate user select dropdown
function populateUserSelect(users) {
    const userSelect = document.getElementById('userSelect');
    userSelect.innerHTML = '<option value="">Choose an affiliate...</option>' + 
        users.map(user => `<option value="${user.id}">${user.first_name} ${user.last_name} (${user.email})</option>`).join('');
}

// Select chat
async function selectChat(chatId) {
    console.log('Selecting chat:', chatId);
    currentChatId = chatId;
    console.log('Current chat ID set to:', currentChatId);
    
    try {
        const response = await fetch(`{{ route("manager.chat.get-chat", ":chatId") }}`.replace(':chatId', chatId));
        const data = await response.json();
        
        if (data.success) {
            displayChat(data.chat);
            loadMessages(chatId);
            subscribeToChat(chatId);
        }
    } catch (error) {
        console.error('Error selecting chat:', error);
    }
}

// Display chat
function displayChat(chat) {
    console.log('Displaying chat:', chat);
    
    const otherParticipant = chat.participants.find(p => p.id !== {{ auth()->id() }});
    
    // Hide no chat selected and show chat messages
    const noChatSelected = document.getElementById('noChatSelected');
    const chatMessages = document.getElementById('chatMessages');
    
    console.log('Before display - noChatSelected classes:', noChatSelected.className);
    console.log('Before display - chatMessages classes:', chatMessages.className);
    
    noChatSelected.classList.add('d-none');
    chatMessages.classList.remove('d-none');
    
    console.log('After display - noChatSelected classes:', noChatSelected.className);
    console.log('After display - chatMessages classes:', chatMessages.className);
    
    document.getElementById('chatUserName').textContent = `${otherParticipant.first_name} ${otherParticipant.last_name}`;
    document.getElementById('chatUserAvatar').src = otherParticipant.avatar_url || 'https://ui-avatars.com/api/?name=' + encodeURIComponent(otherParticipant.first_name + ' ' + otherParticipant.last_name) + '&background=3b82f6&color=fff';
    
    // Update active chat item
    document.querySelectorAll('.chat-item').forEach(item => {
        item.classList.remove('active');
    });
    document.querySelector(`[data-chat-id="${chat.id}"]`).classList.add('active');
    
    // Ensure message input is visible
    setTimeout(() => {
        const messageInput = document.getElementById('messageInput');
        if (messageInput) {
            messageInput.focus();
        }
    }, 100);
}

// Load messages
async function loadMessages(chatId) {
    try {
        const response = await fetch(`{{ route("manager.chat.messages", ":chatId") }}`.replace(':chatId', chatId));
        const data = await response.json();
        
        if (data.success) {
            renderMessages(data.messages);
        }
    } catch (error) {
        console.error('Error loading messages:', error);
    }
}

// Render messages
function renderMessages(messages) {
    const messagesList = document.getElementById('messagesList');
    
    messagesList.innerHTML = messages.map(message => {
        const isOwn = message.user_id === {{ auth()->id() }};
        return `
            <div class="d-flex ${isOwn ? 'justify-content-end' : 'justify-content-start'}">
                <div class="message-bubble ${isOwn ? 'own' : 'other'}">
                    <p class="message-text">${message.message}</p>
                    <small class="message-time">${new Date(message.created_at).toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'})}</small>
                </div>
            </div>
        `;
    }).join('');
    
    // Scroll to bottom
    const messagesContainer = document.getElementById('messagesContainer');
    messagesContainer.scrollTop = messagesContainer.scrollHeight;
}

// Subscribe to chat channel
function subscribeToChat(chatId) {
    if (currentChannel) {
        pusher.unsubscribe(currentChannel.name);
    }
    
    currentChannel = pusher.subscribe(`private-chat.${chatId}`);
    
    currentChannel.bind('message.sent', function(data) {
        if (data.message) {
            addMessageToUI(data.message);
        }
    });
}

// Add message to UI
function addMessageToUI(message) {
    const messagesList = document.getElementById('messagesList');
    if (!messagesList) {
        console.error('Messages list element not found');
        return;
    }
    
    const isOwn = message.user_id === {{ auth()->id() }};
    
    const messageElement = document.createElement('div');
    messageElement.className = `d-flex ${isOwn ? 'justify-content-end' : 'justify-content-start'}`;
    messageElement.innerHTML = `
        <div class="message-bubble ${isOwn ? 'own' : 'other'}">
            <p class="message-text">${message.message}</p>
            <small class="message-time">${new Date(message.created_at).toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'})}</small>
        </div>
    `;
    
    messagesList.appendChild(messageElement);
    
    // Scroll to bottom
    const messagesContainer = document.getElementById('messagesContainer');
    if (messagesContainer) {
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
    }
}

// Send message
async function sendMessage() {
    const messageInput = document.getElementById('messageInput');
    const message = messageInput.value.trim();
    
    console.log('Send message called:', { message, currentChatId });
    
    if (!message) {
        alert('Please enter a message');
        return;
    }
    
    if (!currentChatId) {
        alert('Please select a chat first');
        return;
    }
    
    try {
        const routeUrl = '{{ route("manager.chat.send-message") }}';
        console.log('Sending message to route:', routeUrl);
        const response = await fetch(routeUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({
                chat_id: currentChatId,
                message: message
            })
        });
        
        console.log('Response status:', response.status);
        const data = await response.json();
        console.log('Response data:', data);
        
        if (data.success) {
            console.log('Message sent successfully, clearing input and adding to UI');
            messageInput.value = '';
            // Add message to UI immediately
            addMessageToUI(data.message);
            
            // Focus message input after sending
            setTimeout(() => {
                const messageInput = document.getElementById('messageInput');
                if (messageInput) {
                    messageInput.focus();
                }
            }, 100);
        } else {
            alert('Error: ' + (data.error || 'Failed to send message'));
        }
    } catch (error) {
        console.error('Error sending message:', error);
        alert('Failed to send message: ' + error.message);
    }
}

// Start new chat
async function startNewChat() {
    const userId = document.getElementById('userSelect').value;
    
    if (!userId) return;
    
    try {
        const response = await fetch('{{ route("manager.chat.get-or-create") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({ user_id: userId })
        });
        
        const data = await response.json();
        
        if (data.success) {
            const modal = bootstrap.Modal.getInstance(document.getElementById('startChatModal'));
            modal.hide();
            
            await loadChatData();
            setTimeout(() => {
                selectChat(data.chat.id);
            }, 100);
        } else {
            alert('Error: ' + (data.error || 'Failed to create chat'));
        }
    } catch (error) {
        console.error('Error creating chat:', error);
        alert('Failed to create chat');
    }
}

// Mobile sidebar toggle
function toggleMobileSidebar() {
    const sidebar = document.getElementById('chatSidebar');
    sidebar.classList.toggle('open');
}

// Search functionality
function searchChats(searchTerm) {
    const chatItems = document.querySelectorAll('.chat-item');
    const term = searchTerm.toLowerCase();
    
    chatItems.forEach(item => {
        const name = item.querySelector('.chat-item-name').textContent.toLowerCase();
        const preview = item.querySelector('.chat-item-preview').textContent.toLowerCase();
        
        if (name.includes(term) || preview.includes(term)) {
            item.style.display = 'block';
        } else {
            item.style.display = 'none';
        }
    });
}

// Event Listeners
document.addEventListener('DOMContentLoaded', function() {
    console.log('Chat page loaded');
    
    // Test route availability
    const testRoute = '{{ route("manager.chat.send-message") }}';
    console.log('Send message route:', testRoute);
    
    initializePusher();
    loadChatData();
    
    // Mobile sidebar toggle
    document.getElementById('mobileSidebarToggle').addEventListener('click', toggleMobileSidebar);
    
    // Search functionality
    document.getElementById('chatSearchInput').addEventListener('input', function(e) {
        searchChats(e.target.value);
    });
    
    // Close mobile sidebar when clicking outside
    document.addEventListener('click', function(e) {
        const sidebar = document.getElementById('chatSidebar');
        const toggleBtn = document.getElementById('mobileSidebarToggle');
        
        if (window.innerWidth <= 768 && 
            !sidebar.contains(e.target) && 
            !toggleBtn.contains(e.target) && 
            sidebar.classList.contains('open')) {
            sidebar.classList.remove('open');
        }
    });
    
    // Send message
    document.getElementById('sendButton').addEventListener('click', sendMessage);
    document.getElementById('messageInput').addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            sendMessage();
        }
    });
    
    // Start new chat buttons
    document.getElementById('chatWithUserBtn').addEventListener('click', function() {
        const modal = new bootstrap.Modal(document.getElementById('startChatModal'));
        modal.show();
    });
    
    document.getElementById('startNewChatBtn').addEventListener('click', function() {
        const modal = new bootstrap.Modal(document.getElementById('startChatModal'));
        modal.show();
    });
    
    // User selection
    document.getElementById('userSelect').addEventListener('change', function() {
        const userId = this.value;
        const selectedUserInfo = document.getElementById('selectedUserInfo');
        const startChatBtn = document.getElementById('startChatBtn');
        
        if (userId) {
            // Find user data and display info
            fetch('{{ route("manager.chat.data") }}')
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const user = data.users.find(u => u.id == userId);
                        if (user) {
                            document.getElementById('selectedUserName').textContent = `${user.first_name} ${user.last_name}`;
                            document.getElementById('selectedUserEmail').textContent = user.email;
                            document.getElementById('selectedUserAvatar').src = user.avatar_url || 'https://ui-avatars.com/api/?name=' + encodeURIComponent(user.first_name + ' ' + user.last_name) + '&background=3b82f6&color=fff';
                            selectedUserInfo.style.display = 'block';
                            startChatBtn.disabled = false;
                        }
                    }
                });
        } else {
            selectedUserInfo.style.display = 'none';
            startChatBtn.disabled = true;
        }
    });
    
    // Start chat button
    document.getElementById('startChatBtn').addEventListener('click', startNewChat);
});
</script>
@endpush
